import six

import shutil

import tarfile
import tempfile
from collections import defaultdict

import typing as tp

import json
import logging
import os
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.devtools.ChangesDetector import resources
import sandbox.common.types.misc as ctm

TASK_LOG_FORMAT = "%(asctime)s (%(delta)10.3fs) [%(processName)s:%(threadName)s] %(levelname)-6s (%(name)s) %(message)s"


autocheck_configs = [
    "linux",
    "mandatory",
    "sanitizers",
    "gcc-msvc-musl",
    "ios-android-cygwin",
]


class ChangesDetector(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024 * 5
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 16 * 1024, None)

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    logger = logging.getLogger("ChangesDetector")

    class Parameters(sdk2.Task.Parameters):
        description = 'Task for detect changes between revisions'
        kill_timeout = 70 * 60

        with sdk2.parameters.RadioGroup("Build profile") as build_profile:
            for name, value in [("Pre commit", "pre_commit"), ("Post commit", "post_commit")]:
                build_profile.values[value] = build_profile.Value(name, default=value == 'pre_commit')

        targets = sdk2.parameters.List("Targets")  # type: list[str]

        with sdk2.parameters.Group("Build"):
            arc_url_left = sdk2.parameters.ArcadiaUrl("Left Arc url")
            patch_left = sdk2.parameters.Url("Left Patch url")

            arc_url_right = sdk2.parameters.ArcadiaUrl("Right Arc url")
            patch_right = sdk2.parameters.Url("Right Patch url")

            with sdk2.parameters.CheckGroup("Platforms") as platforms:
                platforms.choices = [(_, _) for _ in autocheck_configs]

            generate_left_right = sdk2.parameters.Bool(
                'Generate left-right graph instead right-left', default=False)  # type: bool

            autocheck_yaml_url = sdk2.parameters.ArcadiaUrl('Url for autocheck.yaml', default=None)

            ymake_cache_kind = sdk2.parameters.String("Ymake cache kind for download", default='parser_json')  # type: str
            use_ymake_cache = sdk2.parameters.String("Ymake cache kind to use", default='parser_json')  # type: str
            use_ymake_minimal_cache = sdk2.parameters.String('Ymake cache kind to use in minimal cache graph')  # type: str
            use_imprint_cache = sdk2.parameters.Bool('use_imprint_cache', default=True)  # type: bool
            trust_cache_fs = sdk2.parameters.Bool('trust_cache_fs', default=True)  # type: bool

            do_not_execute_metagraph = sdk2.parameters.Bool(
                "Do not execute metagraphs (use only for acceptance tests)",
                default=False,
            )

        use_caches = sdk2.parameters.Bool("Use caches", default=True)

        with sdk2.parameters.Group("Custom ya execution options"):
            distbuild_pool = sdk2.parameters.String("DEPRECATED", default=None)
            gg_distbuild_pool = sdk2.parameters.String("Graph Generation distbuild pool", default="autocheck/gg/postcommits/tier0_discovery")

        with sdk2.parameters.Group("Distbuild priority options"):
            autocheck_revision = sdk2.parameters.Integer("Autocheck revision", description="Used for calculate priority")

        with sdk2.parameters.Group("Postprocessing"):
            compress_results = sdk2.parameters.Bool("Compress results", default=False)
            affected_targets = sdk2.parameters.Bool("Store affected targets to resource", default=True)
            configure_errors = sdk2.parameters.Bool("Store configure errors to resource", default=False)
            meta_graphs = sdk2.parameters.Bool("Store metagraphs to resource", default=False)

        with sdk2.parameters.Group("Task execution"):
            ya_token = sdk2.parameters.YavSecret('Yav secret to update resources')
            use_ram_disk = sdk2.parameters.Bool("Use ramdisk for working directory", default=True)
            logs_resource_ttl = sdk2.parameters.Integer("Logs resource ttl", default=30)
            preburn_tools = sdk2.parameters.Bool("Preburn tools", description="Preload some tools to prevent multiple downloads. see DEVTOOLS-9156", default=True)
            store_core_dump = sdk2.parameters.Bool("Store coredumps", description="Use sandbox subprocess runner", default=True)  # TODO: Disable after YA-64

            binary_release = binary_task.binary_release_parameters(stable=True)

    def on_enqueue(self):
        if self.Parameters.do_not_execute_metagraph:
            assert not self.Parameters.affected_targets
            assert not self.Parameters.configure_errors

        if self.Parameters.use_ram_disk:
            ram_disk_size = 16
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, ram_disk_size * 1024, None)
            self.Requirements.disk_space = 4 * 1024
        else:
            self.Requirements.disk_space = 10 * 1024
            self.Requirements.ramdrive = None

    def _create_repos(self, repository_config, left_working_copy, right_working_copy, autocheck_yaml_working_copy):
        from yalibrary.ggaas.bare_repository import ArcThinArcadiaRepository
        from yalibrary.ggaas.task_manager import ThreadTaskManager

        work_dir = os.path.join(self.work_dir, 'arc_repos')
        arc_repo_logs = os.path.join(self.results_dir, 'arc_repos')

        left_arc = ArcThinArcadiaRepository(left_working_copy, repository_config=repository_config,
                                            work_dir=os.path.join(work_dir, 'left'),
                                            logs_dir=os.path.join(arc_repo_logs, 'left'))
        right_arc = ArcThinArcadiaRepository(right_working_copy, repository_config=repository_config,
                                             work_dir=os.path.join(work_dir, 'right'),
                                             logs_dir=os.path.join(arc_repo_logs, 'right'))
        autocheck_yaml_repo = None
        if autocheck_yaml_working_copy:
            autocheck_yaml_repo = ArcThinArcadiaRepository(autocheck_yaml_working_copy,
                                                           repository_config=repository_config,
                                                           work_dir=os.path.join(work_dir, 'autocheck_yaml'),
                                                           logs_dir=os.path.join(arc_repo_logs, 'autocheck_yaml'))

        with ThreadTaskManager() as tasks:
            _task_left = tasks.create_task("Create left arc repo", left_arc.setup)
            _task_right = tasks.create_task("Create right arc repo", right_arc.setup)
            if autocheck_yaml_repo:
                _task_autocheck_yaml = tasks.create_task(
                    "Create autocheck.yaml repo", autocheck_yaml_repo.setup)

            # Will be automatically join

        return left_arc, right_arc, autocheck_yaml_repo

    def _setup_logging(self):
        logging.basicConfig(format=TASK_LOG_FORMAT)

    def on_execute(self):
        if six.PY2:
            from contextlib2 import ExitStack
        else:
            from contextlib import ExitStack

        with ExitStack() as stack:
            if self.Parameters.store_core_dump:
                self.logger.info("Enable store coredump")

                stack.enter_context(sdk2.helpers.ProcessRegistry)

                def run(*args, **kwargs):
                    try:
                        return sdk2.helpers.process.subprocess.run(*args, **kwargs)
                    except sdk2.helpers.process.subprocess.CalledProcessError as e:
                        # Use default exception, because code didn't understand this class
                        from yalibrary.ya_helper.common.run_subprocess import CalledProcessError
                        import sys
                        _, _, _traceback = sys.exc_info()
                        _e = CalledProcessError(e.returncode, e.cmd, e.output, e.stderr)
                        _e.__dict__ = e.__dict__
                        six.reraise(type(_e), _e, _traceback)

                from yalibrary.ya_helper.common.run_subprocess import set_default_runner

                set_default_runner(run)

            self._run()

    def _run(self):
        # Prepare section
        self.log_resource.ttl = self.Parameters.logs_resource_ttl

        # Imports sections
        from core.profiler import profile_step_started, profile_step_finished

        from yalibrary.ggaas.bare_repository import ArcThinArcadiaRepository, RepositoryConfig

        from yalibrary.ggaas.interface.repo import WorkingCopy, PatchURL

        from yalibrary.ggaas.interface.graph_request import GraphExecuteRequest, GraphSubtractBuildRequest

        from yalibrary.ya_helper.common import make_folder, GSIDParts

        from yalibrary.ggaas.task_manager import ProcessTaskManager
        from yalibrary.ggaas.interface.execute import ExecuteOptions, YaCustomExecuteOptions
        from yalibrary.ggaas.interface.build import BuildOptions, Targets, SubtractBuildForPlatform

        from yalibrary.ggaas.sandbox import ProcessRequestSandbox, DoExecutePartitionSandbox, preburn_tools

        from yalibrary.platform_matcher.autocheck_configuration import ConfigurationParser, configurations as default_configurations

        DoExecutePartitionSandbox.TASK_LOG_FORMAT = TASK_LOG_FORMAT

        # TODO: Use stages for statistics

        # __init__ section
        self.results_dir = str(self.log_path("changes_detector"))
        make_folder(self.results_dir)
        self.work_dir = None

        if self.Parameters.use_ram_disk:
            self.work_dir = str(self.ramdrive.path)
        else:
            self.work_dir = str(self.log_path('changes_detector_tmp'))

        self.logger.debug("Working directory: %s", self.work_dir)

        self.left_arc = None  # type: tp.Optional[ArcThinArcadiaRepository]
        self.right_arc = None  # type: tp.Optional[ArcThinArcadiaRepository]
        self.autocheck_yaml_repo = None  # type: tp.Optional[ArcThinArcadiaRepository]

        self.tasks = ProcessTaskManager()
        self.request_processor = None  # type:
        self.execution_result_resource = None  # type: tp.Optional[sdk2.Resource]
        self.execution_result_data = None  # type: tp.Optional[sdk2.ResourceData]

        self.logger.info("PREPARE VALUES")
        build_profile = BuildOptions.BuildProfile(self.Parameters.build_profile)

        tokens = self.Parameters.ya_token.data()
        ya_token = tokens['YA_TOKEN']

        # Create repos
        self.logger.info("CREATE REPOS")
        with RepositoryConfig(work_dir=os.path.join(self.work_dir, 'repository_config'), token=ya_token) as repository_config:
            del ya_token  # Just for safety

            profile_step_started('create_repos')
            self.left_arc, self.right_arc, self.autocheck_yaml_repo = self._create_repos(
                repository_config,
                left_working_copy=WorkingCopy(
                    str(self.Parameters.arc_url_left),
                    patch=PatchURL(str(self.Parameters.patch_left)) if self.Parameters.patch_left else None
                ),
                right_working_copy=WorkingCopy(
                    str(self.Parameters.arc_url_right),
                    patch=PatchURL(str(self.Parameters.patch_right)) if self.Parameters.patch_right else None
                ),
                autocheck_yaml_working_copy=WorkingCopy(
                    str(self.Parameters.autocheck_yaml_url)
                ) if self.Parameters.autocheck_yaml_url else None
            )
            profile_step_finished('create_repos')

            # Preload Ya into tools dir
            profile_step_started('preburn_tools')
            # TODO: Back to the repo-based tool dir when failed
            if self.Parameters.preburn_tools:
                preburn_tools(self.left_arc, os.path.join(os.path.join(self.work_dir, "preload_tools")))
            profile_step_finished('preburn_tools')

            profile_step_started('receive_configuration')
            if self.autocheck_yaml_repo:
                self.logger.info("Using autocheck.yaml from %s", self.autocheck_yaml_repo)
                autocheck_yaml_path = os.path.join(self.autocheck_yaml_repo.source_root, "autocheck/autocheck.yaml")

                autocheck_yaml_content = open(autocheck_yaml_path, "rt").read()

                config_parser = ConfigurationParser(
                    source=autocheck_yaml_content
                )
                configurations = dict(config_parser.parse())
            else:
                self.logger.info("Using built-in autocheck.yaml")
                configurations = default_configurations()
            profile_step_finished('receive_configuration')

            _is_exc = False
            requests = []

            for name in self.Parameters.platforms:
                configuration = configurations[name]

                for partition in configuration.all_partitions():
                    build_request = SubtractBuildForPlatform(
                        configuration=configuration,
                        partition_index=partition.partition_index,
                        partitions_count=configuration.partitions_count,
                        generate_left_right=self.Parameters.generate_left_right
                    )

                    request = GraphSubtractBuildRequest(
                        left_arc_repo=self.left_arc, right_arc_repo=self.right_arc,
                        build_requests=build_request,
                        targets=Targets(*self.Parameters.targets),
                        build_options=BuildOptions(
                            build_profile=build_profile,
                            priority_base=self.Parameters.autocheck_revision or self.id,  # Using id for priority calculation where is no revision
                            ymake_cache_kind=self.Parameters.ymake_cache_kind,
                            use_ymake_cache=self.Parameters.use_ymake_cache,
                            use_ymake_minimal_cache=self.Parameters.use_ymake_minimal_cache,
                            use_imprint_cache=self.Parameters.use_imprint_cache,
                            trust_cache_fs=self.Parameters.trust_cache_fs,)
                    )

                    if not self.Parameters.do_not_execute_metagraph:
                        gg_full_distbuild_path = os.path.join(
                            partition.graph_generation_pool_tree,
                            self.Parameters.gg_distbuild_pool,
                        )

                        request = GraphExecuteRequest(
                            graph_request=request,
                            execute_options=ExecuteOptions(
                                ya_custom_options=YaCustomExecuteOptions(
                                    distbs_pool=gg_full_distbuild_path,
                                    coordinators_filter=partition.graph_generation_coordinators_filter,
                                    make_context_on_distbuild_requirements=partition.graph_generation_requirements,
                                ),
                                build_profile=build_profile,
                                gsid_parts=GSIDParts.create_from_environ()
                            ),
                        )

                    requests.append(request)

            try:
                self.request_processor = ProcessRequestSandbox(
                    requests,
                    self.results_dir,
                )

                self.logger.info("WAITING FOR PROCESSING AND COLLECT RESULTS")

                postprocessing_tasks = []

                self.execution_result_resource = self._create_result_resource()

                self.execution_result_data = sdk2.ResourceData(self.execution_result_resource)
                self.execution_result_data.path.mkdir()

                for processing_result, build_for in self.request_processor.process(use_caches=self.Parameters.use_caches):
                    if processing_result is None:
                        self.logger.info("Result from %s is None, skip", build_for)
                        continue

                    postprocessing_tasks.append(
                        self.tasks.create_task(
                            "Postprocessing for {}".format(build_for.about),
                            self.postprocessing,
                            processing_result, build_for
                        ))
            except Exception:
                _is_exc = True
                self.logger.exception("While process %s", request)
                raise
            finally:
                if self.left_arc:
                    self._run_du(self.left_arc.work_dir)
                    self.left_arc.finalize(remove_folders=not _is_exc)

                if self.right_arc:
                    self._run_du(self.right_arc.work_dir)
                    self.right_arc.finalize(remove_folders=not _is_exc)

                if self.autocheck_yaml_repo:
                    self._run_du(self.autocheck_yaml_repo.work_dir)
                    self.autocheck_yaml_repo.finalize(remove_folders=not _is_exc)

                self._run_du(repository_config.work_dir)

            affected_targets = defaultdict(set)
            configure_errors = defaultdict(lambda: defaultdict(list))
            meta_graphs = []

            processed_results = []

            for result in self.tasks.gather_async(*postprocessing_tasks, empty_ok=True):
                configuration_name = result['configuration_name']
                processed_results.append(result['processed_result'])

                if self.Parameters.affected_targets:
                    with open(result['affected_targets'], 'rt') as f:
                        for item in f.readlines():
                            if not item:
                                continue
                            affected_targets[configuration_name].add(item.strip())

                if self.Parameters.configure_errors:
                    with open(result['configure_errors'], 'rt') as f:
                        errors_from_partition = json.load(f)

                    errors_result = configure_errors[configuration_name]

                    for k, v in six.iteritems(errors_from_partition):
                        errors_result[k].extend(v)

                if self.Parameters.meta_graphs:
                    meta_graphs.append(result)

        affected_targets = {k: sorted(v) for k, v in six.iteritems(affected_targets)}

        self.logger.info("STORE TO RESOURCE")

        if self.Parameters.affected_targets:
            self.store_resource(
                resources.AffectedTargets,
                "Affected targets", "affected_targets.json",
                affected_targets
            )

        if self.Parameters.configure_errors:
            self.store_resource(
                resources.ConfigureErrors,
                "Configure errors", "configure_errors.json",
                configure_errors
            )

        if self.Parameters.do_not_execute_metagraph and self.Parameters.meta_graphs:
            from sandbox.projects.autocheck.lib.core.resources import YaMakeMetaGraphResult
            for item in meta_graphs:
                resource = YaMakeMetaGraphResult(
                    task=self,
                    correct_order=True,
                    **item['resource_kwargs']
                )

                logging.info("Resource id for store %s: %s", resource.id)

                resource_data = sdk2.ResourceData(resource)

                try:
                    resource_data.path.mkdir(mode=0o755, parents=True, exist_ok=False)

                    for name, path in six.iteritems(item['meta_graphs']):
                        shutil.copy(
                            path, str(resource_data.path / "{}.json".format(name))
                        )
                except:
                    self.logger.exception("While preparing meta graph")
                    resource_data.broken()
                else:
                    resource_data.ready()

        for item in processed_results:
            try:
                self._move_results(item)
            except:
                self.logger.exception("While moving %s", item)

        if self.Parameters.do_not_execute_metagraph:
            self.execution_result_data.broken()
        elif self.execution_result_data is not None:
            self.execution_result_data.ready()

        self.logger.info("Good work fella")

    def _create_result_resource(self):
        existed_resources_count = len(
            list(resources.ChangesDetectorExecutionResults.find(task=self).limit(1000)))

        return resources.ChangesDetectorExecutionResults(
            task=self,
            description="Subtraction result for {}".format(self.id),
            path="big_results_{}".format(existed_resources_count + 1),
            ttl=14,
        )

    def _run_du(self, path):
        import subprocess

        try:
            p1 = subprocess.Popen(['find', path, '-type', 'f', '-exec', 'du', '-ah', '{}', '+'], stdout=subprocess.PIPE)
            p2 = subprocess.Popen(['grep', '-v', '/$'], stdin=p1.stdout, stdout=subprocess.PIPE)
            p3 = subprocess.Popen(['sort', '-rh'], stdin=p2.stdout, stdout=subprocess.PIPE)
            p4 = subprocess.Popen(['head', '-20'], stdin=p3.stdout, stdout=subprocess.PIPE)

            result, _ = p4.communicate()

            self.logger.debug("Biggest files in `%s`: \n%s", path, six.ensure_text(result))
        except Exception:
            self.logger.exception("While running du in %s", path)

    def store_resource(self,
                       resource_class,
                       description, path,
                       json_data):
        from exts.fs import ensure_removed

        self.logger.info("Store `%s` to resource", description)

        full_path = "{}{}".format(path, '.tar.gz' if self.Parameters.compress_results else '')

        ensure_removed(full_path)

        resource = resource_class(
            task=self,
            description=description,
            path=full_path,
            ttl=30
        )

        tmp_file_name = tempfile.mktemp(dir=self.work_dir)

        with open(tmp_file_name, 'wt') as f:
            json.dump(json_data, f, sort_keys=True)

        data = sdk2.ResourceData(resource)

        if self.Parameters.compress_results:
            self.logger.debug("Compressing data from %s", tmp_file_name)
            with tarfile.open(str(data.path), 'w:gz') as tar:
                tar.add(tmp_file_name, arcname=path)
        else:
            shutil.copy(tmp_file_name, str(data.path))

        data.ready()
        self.logger.info("Saved `%s` to resource: %d", description, resource.id)

    def postprocessing(self, processing_result, build_for):
        # type: (BaseGraphResult, BuildForPlatform) -> dict
        from core.profiler import profile_step_started, profile_step_finished
        from yalibrary.ggaas.interface.execute import GraphsSubtractExecutionResult
        from yalibrary.ggaas.interface.build import GraphsSubtractResult

        step_name = "{}:postprocess".format(build_for.about)
        profile_step_started(step_name)

        if self.Parameters.do_not_execute_metagraph:
            assert isinstance(processing_result, GraphsSubtractResult)
            results = self._process_build(build_for, processing_result, step_name)
        else:
            assert isinstance(processing_result, GraphsSubtractExecutionResult)
            # TODO: Store metagraphs here too
            # TODO: Add GraphsSubtractResult into GraphsSubtractExecutionResult
            results = self._process_execution(build_for, processing_result, step_name)

        profile_step_finished(step_name)

        return results

    def _move_results(self, processing_result):
        # type: (BaseGraphResult) -> None
        assert self.execution_result_data is not None

        from yalibrary.ggaas.interface.execute import GraphsSubtractExecutionResult

        if isinstance(processing_result, GraphsSubtractExecutionResult):
            _from = processing_result.result_dir
            relpath = os.path.relpath(_from, self.results_dir)
            _to = os.path.join(str(self.execution_result_data.path), relpath)
            self.logger.debug("Move from %s to %s",
                              _from, _to)
            shutil.move(
                _from,
                _to,
            )
            with open(os.path.join(_from), "wt") as f:
                f.write("Folder {} was moved into {}/{}".format(
                    _from,
                    self.execution_result_resource.http_proxy,
                    relpath
                ))

    def _process_build(self, build_for, build_result, step_name):
        # type: (BuildForPlatform, GraphsSubtractResult, str) -> dict
        from yalibrary.platform_matcher import configurations

        autocheck_config_path = configurations()[build_for.configuration.name].config_path
        targets = ';'.join(self.Parameters.targets)
        partition_index = build_for.partition_index
        partitions_count = build_for.partitions_count
        left_rev, left_url, left_patch = self.left_arc.reference.svn_revision, self.left_arc.reference.like_autocheck, self.Parameters.patch_left
        right_rev, right_url, right_patch = self.right_arc.reference.svn_revision, self.right_arc.reference.like_autocheck, self.Parameters.patch_right

        if len(self.Parameters.targets) == 1 and self.Parameters.targets[0] == "autocheck":
            circuit_type = "full"
        else:
            circuit_type = "fast"

        desc = 'Meta Graphs {} ({} {} {}) ({} {} {}) {} {}/{}'.format(
            circuit_type,
            left_rev, left_url, left_patch,
            right_rev, right_url, right_patch,
            targets,
            partition_index, partitions_count,
        )

        logging.debug("Resource description for store meta graphs: %s",
                      desc)

        resource_kwargs = dict(
            description=desc,
            path='meta_graphs_{}_{}'.format(partition_index, partitions_count),
            left_url=left_url,
            left_patch=left_patch,
            right_url=right_url,
            right_patch=right_patch,
            circuit_type=circuit_type,
            is_dummy_arcadia=targets == 'devtools/dummy_arcadia/empty_project',
            targets=targets,
            autocheck_config_path=autocheck_config_path,
            partition_index=partition_index,
            partitions_count=partitions_count,
            kind=str(self.Parameters.binary_executor_release_type),
            source=str(self.type),
        )

        if left_rev is not None:
            resource_kwargs['left_revision'] = left_rev
        if right_rev is not None:
            resource_kwargs['right_revision'] = right_rev

        return {
            'configuration_name': build_for.configuration.name,
            'resource_kwargs': resource_kwargs,
            'processed_result': build_result,

            'meta_graphs': {
                'left': build_result.left.meta_graph_path,
                'right': build_result.right.meta_graph_path,
                'subtract': build_result.subtracting_context_result,
            }
        }

    def _process_execution(self, build_request, execution_result, step_name):
        results = {
            'configuration_name': build_request.configuration.name,
            'processed_result': execution_result,
            'affected_targets': None,
            'configure_errors': None,
            'graph': None,
        }
        from yalibrary.ggaas.graph_parser import Graph
        from core.profiler import profile_step_started, profile_step_finished

        graph = Graph(execution_result.context_cut_right_left_compressed,
                      execution_result.affected_uids_right_left if self.Parameters.affected_targets else None,
                      self.Parameters.configure_errors)

        profile_step_started(step_name + ":graph_parse")
        graph.process()
        results['graph'] = execution_result.context_cut_right_left_compressed
        profile_step_finished(step_name + ":graph_parse")

        if self.Parameters.affected_targets:
            profile_step_started(step_name + ":affected_targets")

            affected_targets_path = os.path.join(execution_result.result_dir, 'affected_targets.txt')

            with open(affected_targets_path, "wt") as f:
                for item in graph.affected_targets():
                    f.write(item)
                    f.write("\n")
            results['affected_targets'] = affected_targets_path
            profile_step_finished(step_name + ":affected_targets")

        if self.Parameters.configure_errors:
            profile_step_started(step_name + ":configure_errors")

            configure_errors_path = os.path.join(execution_result.result_dir, 'configure_errors.json')
            with open(configure_errors_path, 'wt') as f:
                json.dump(graph.configure_errors, f)
            results['configure_errors'] = configure_errors_path

            profile_step_finished(step_name + ":configure_errors")

        return results

    def _shutdown_all(self):
        if hasattr(self, "tasks"):
            if self.tasks:
                self.tasks.gracefully_shutdown()

    def on_timeout(self, prev_status):
        self._shutdown_all()

    def on_terminate(self):
        self._shutdown_all()

    def on_finish(self, prev_status, status):
        if hasattr(self, "tasks"):
            if self.tasks:
                self.tasks.join()
