import datetime

import json
import logging
import math
from itertools import product

import dateutil.parser

from sandbox import sdk2
import sandbox.projects.common.binary_task as binary_task
import sandbox.common.types.task as ctt
import sandbox.projects.sandbox.deploy_binary_task as dbt
import sandbox.projects.devtools.ChangesDetector as cd
from yalibrary.platform_matcher import configurations
from yalibrary.ggaas.graph_parser import Graph


def process_custom_fields(normal_dict: dict):
    custom_fields = []
    for name, value in normal_dict.items():
        custom_fields.append({
            'name': name,
            'value': value,
        })

    return custom_fields


class ChangesDetectorAcceptance(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Used for test ChangesDetector task
    https://a.yandex-team.ru/arc_vcs/ci/registry/projects/devtools/ggaas/changes_detector_acceptance.yaml
    """
    class Requirements(sdk2.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.Url('Arcadia url')
        arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent)', multiline=True)

        ya_arc_secret = sdk2.parameters.YavSecret('Yav secret for CHANGES_DETECTOR with YA_TOKEN inside', default="sec-01ffhqcd3rg8bd48fjdc2mwyqr")

        with sdk2.parameters.RadioGroup("Deploy task:") as deploy_task:
            deploy_task.values["changes_detector"] = deploy_task.Value("ChangesDetector", default=True)
            deploy_task.values["changes_detector_acceptance"] = deploy_task.Value("ChangesDetectorAcceptance")

        with deploy_task.value["changes_detector"]:
            precommit_run = sdk2.parameters.Bool("Is acceptance run for precommit", default=True)
            check_with_abp2 = sdk2.parameters.Bool("Check with graphs from AutocheckBuildParent2 tasks", default=False)

        just_build = sdk2.parameters.Bool("Just build resource", default=False)

        release_run = sdk2.parameters.Bool("Is acceptance run for release (postcommit only!)", default=False)

        # TODO: Split release: build -> gg -> exec for affected_targets + exec for configuration_error -> prestable -> stable

        binary_release = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Context):
        _reference_task = None
        dbt_task_id = None
        cd_task_id = None
        dbt_acceptance_id = None
        dbt_abp2_id = None
        dbt_cd_id = None

    def on_execute(self):
        super(ChangesDetectorAcceptance, self).on_execute()
        self.logger = logging.getLogger(__name__)

        if self.Parameters.deploy_task == "changes_detector":
            if self.Parameters.precommit_run and self.Parameters.release_run:
                raise ValueError("You cannot run this task in precommit with release_run")

            if self.Parameters.precommit_run:
                self.logger.info("Running in precommit mode")
            else:
                self.logger.info("Running in postcommit mode")

        if self.Parameters.just_build:
            self.logger.info("Just build the resource")
            self._do_just_build()

        if self.Parameters.release_run:
            self.logger.info("Running in release mode")

        if self.Parameters.deploy_task == 'changes_detector':
            if self.Parameters.check_with_abp2:
                self.logger.info("Check graphs from CHANGES_DETECTOR with AUTOCHECK_BUILD_PARENT_2")
                self._do_check_cd_with_abp2()
            else:
                self.logger.info("Check CHANGES_DETECTOR")
                self._do_check_changes_detector()
        else:
            self.logger.info("Check CHANGES_DETECTOR_ACCEPTANCE")
            self._do_check_changes_detector_acceptance()

    # TODO: Extract processing to different classes

    # AUTOCHECK_BUILD_PARENT2 <-> CHANGES_DETECTOR graphs
    def _do_check_cd_with_abp2(self):
        with self.memoize_stage.create_abp2_tasks(commit_on_entrance=False):
            task_infos = self._search_abp2()
            self.Context.dbt_abp2_id = self._create_dbt_for_abp2_tasks(task_infos)
            self.Context.dbt_cd_id = self._create_dbt_for_cd_tasks(task_infos)

            raise sdk2.WaitTask(
                [self.Context.dbt_abp2_id, self.Context.dbt_cd_id],
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
                wait_all=True,
            )

        dbt_abp2_info = self.server.task[self.Context.dbt_abp2_id].read()
        self.logger.debug("DBP for ABP2 info: %s", dbt_abp2_info)

        if dbt_abp2_info['status'] != "SUCCESS":
            raise ValueError(f"Wrong status for DEPLOY_BINARY_TASK {dbt_abp2_info['id']}")

        dbt_cd_info = self.server.task[self.Context.dbt_cd_id].read()
        self.logger.debug("DBP for CD info: %s", dbt_cd_info)

        if dbt_cd_info['status'] != "SUCCESS":
            raise ValueError(f"Wrong status for DEPLOY_BINARY_TASK {dbt_cd_info['id']}")

        self.logger.info("Control is AUTOCHECK_BUILD_PARENT2, test is CHANGES_DETECTOR")

        self.set_info("Control is AUTOCHECK_BUILD_PARENT_2 resources\n"
                      "Test is CHANGES_DETECTOR resources")

        self._compare_all_graphs(
            dbt_abp2_info,
            dbt_cd_info,
            deep=1,
        )

        return True

    def _search_abp2(self):
        task_infos = []
        expected = 0
        found = 0

        for config_name in configurations().keys():
            for circuit_type in ('fast', 'full'):
                expected += 1
                self.logger.info("Search tasks for %s / %s", config_name, circuit_type)
                # TODO: Don't search beta-testers
                query = dict(
                    type="AUTOCHECK_BUILD_PARENT_2", status="SUCCESS", owner="AUTOCHECK_TRUNK_PRECOMMIT", order="-id",
                    input_parameters={
                        'autocheck_config_name': config_name,
                        'is_precommit': True,
                        'save_meta_graphs': True,
                        'stress_test': False,
                        'acceptance_task': False,
                        'circuit_type': circuit_type,
                        'use_in_compare_repo_state': True,
                        'custom_targets_list': "[]",
                    }, limit=1, hidden=True,
                )
                tasks = self.server.task.read(**query)["items"]
                if not len(tasks):
                    self.logger.warning("Can't found tasks for %s / %s", config_name, circuit_type)
                    self.logger.debug("Query: %s", query)
                    continue

                task_info = tasks[0]
                self.logger.debug("Found %s", task_info)

                if (datetime.datetime.now(datetime.timezone.utc) -
                    dateutil.parser.parse(task_info['execution']['finished'])) > datetime.timedelta(hours=6):
                    self.logger.warning(
                        "Something goes wrong: last found task "
                        f"for %s / %s finished more than 6 hours later. "
                        "Update search query in ChangesDetectorAcceptance task, rebuild and "
                        "update stable resource in "
                        "https://a.yandex-team.ru/arc_vcs/ci/registry/projects/devtools/ggaas/changes_detector_acceptance.yaml "
                        "OR maybe autocheck.yaml has been changed", config_name, circuit_type)
                    continue

                task_infos.append(task_info)
                found += 1

        if found < expected / 2:
            raise ValueError(f"Too few elements: found {found}, expected {expected}; see warnings for more info")

        return task_infos

    def _create_dbt_for_abp2_tasks(self, task_infos):
        description = "[GGaaS auto acceptance]\n" \
                      "Check AutocheckBuildParent2 and ChangesDetector consistence"

        integration_payload_base = {
            "use_streaming": False,
            "report_to_ci": False,
            "acceptance_task": True,
            "stop_after_meta_graph": True,
            'binary_executor_release_type': 'custom',
            'ya_bin_token': "YA_TOKEN:GGAAS",
            'arc_token': "ARC_TOKEN:GGAAS",
            'semaphores': json.dumps([]),
            'send_logs_to_logbroker': False,
            'autocheck_config_revision': self.Parameters.arcadia_url.split("#")[-1],  # This is too bad but I'm tired so
        }

        integration_payloads = [
            {**task_info['input_parameters'], **integration_payload_base}
            for task_info in task_infos
        ]

        resource_attrs = {
            'released': 'cancelled',
            'task_type': 'AUTOCHECK_BUILD_PARENT_2',
            'taskbox_enabled': True,
        }

        deploy_binary_task_parameters = dict(
            arcadia_url=self.Parameters.arcadia_url,
            target='sandbox/projects/autocheck/AutocheckBuildParent2',
            arcadia_patch=self.Parameters.arcadia_patch,
            attrs=resource_attrs,
            integrational_check=not self.Parameters.just_build,
            integrational_check_payload=[{
                'type': "AUTOCHECK_BUILD_PARENT_2",
                'custom_fields': process_custom_fields(integration_payload),
            } for integration_payload in integration_payloads],
            use_arc=True,
            arc_oauth_token=str(self.Parameters.ya_arc_secret),
            resource_for_parent=self.Parameters.release_run,
            binary_executor_release_type="custom",
            taskbox_enabled="on",
        )

        req = deploy_binary_task_parameters.get('__requirements__', {})
        req['tasks_resource'] = self.Requirements.tasks_resource
        deploy_binary_task_parameters.update(__requirements__=req)

        task_dbt = dbt.DeployBinaryTask(
            self,
            description=description,
            **deploy_binary_task_parameters,
        )
        task_dbt.enqueue()
        self.logger.debug("DeployBinaryTask: {}".format(task_dbt.id))
        return task_dbt.id

    def _create_dbt_for_cd_tasks(self, task_infos):
        description = "[GGaaS auto acceptance]" \
                      "Check AutocheckBuildParent2 and ChangesDetector consistence"

        integration_payloads = []
        for task_info in task_infos:
            params = task_info['input_parameters']

            platform = params['autocheck_config_name']

            integration_payload = {
                'build_profile': 'pre_commit',  # is_precommit
                'targets': params['autocheck_make_only_dirs'].split(";"),
                'arc_url_left': params['checkout_arcadia_from_url_for_in_compare_repo_state'],
                'patch_left': params['arcadia_patch_for_in_compare_repo_state'],
                'arc_url_right': params['checkout_arcadia_from_url'],
                'patch_right': params['arcadia_patch'],
                'platforms': [platform],
                'do_not_execute_metagraph': True,
                'use_caches': True,
                'distbuild_pool': f"//sas_gg/{params['pool_name']}",
                'compress_results': False,
                'affected_targets': False,
                'configure_errors': False,
                'meta_graphs': True,
                'ya_token': str(self.Parameters.ya_arc_secret),
                'use_ram_disk': True,
                'logs_resource_ttl': 2,
                'binary_executor_release_type': 'custom',
                'generate_left_right': params['build_left_right_graph']
            }

            integration_payloads.append(integration_payload)

        attrs = {
            'released': 'cancelled',
            'task_type': 'CHANGES_DETECTOR',
            'taskbox_enabled': True,
        }

        deploy_binary_task_parameters = dict(
            arcadia_url=self.Parameters.arcadia_url,
            target='sandbox/projects/devtools/ChangesDetector/bin',
            arcadia_patch=self.Parameters.arcadia_patch,
            attrs=attrs,
            integrational_check=not self.Parameters.just_build,
            integrational_check_payload=[{
                'type': "CHANGES_DETECTOR",
                'custom_fields': process_custom_fields(integration_payload),
            } for integration_payload in integration_payloads],
            use_arc=True,
            arc_oauth_token=str(self.Parameters.ya_arc_secret),
            resource_for_parent=self.Parameters.release_run,
            binary_executor_release_type="custom",
            taskbox_enabled="on",
        )

        req = deploy_binary_task_parameters.get('__requirements__', {})
        req['tasks_resource'] = self.Requirements.tasks_resource
        deploy_binary_task_parameters.update(__requirements__=req)

        task_dbt = dbt.DeployBinaryTask(
            self,
            description=description,
            **deploy_binary_task_parameters,
        )
        task_dbt.enqueue()
        self.logger.debug("DeployBinaryTask: %s", task_dbt.id)
        return task_dbt.id

    # CHANGES_DETECTOR_ACCEPTANCE check
    def _do_check_changes_detector_acceptance(self):
        # Create all tasks with all parameters
        with self.memoize_stage.create_acceptance_tasks(commit_on_entrance=False):
            self.Context.dbt_acceptance_id = self._create_acceptance_tasks()

            raise sdk2.WaitTask(
                [self.Context.dbt_acceptance_id],
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
                wait_all=True,
            )

        dbt_info = self.server.task[self.Context.dbt_acceptance_id].read()
        self.logger.debug("Stable CHANGES_DETECTOR data: %s", dbt_info)
        if dbt_info['status'] != "SUCCESS":
            raise ValueError("Wrong status for DEPLOY_BINARY_TASK task {}".format(dbt_info['id']))

    def _create_acceptance_tasks(self):
        self.logger.info("Create acceptance tasks")
        description = "[GGaaS auto acceptance]\n" \
                      "Check ChangesDetectorAcceptance task"

        integration_payload_base = {
            "arcadia_url": self.Parameters.arcadia_url,
            "ya_arc_secret": str(self.Parameters.ya_arc_secret),
            "deploy_task": "changes_detector",
            "release_run": False,
            'binary_executor_release_type': 'custom',
        }

        integration_payloads = [
            {**integration_payload_base, "precommit_run": precommit_run, "check_with_abp2": check_with_abp2}
            for precommit_run, check_with_abp2 in product(
                (True, False),
                (False, True),
            )]

        attrs = {
            'released': 'unstable',
            'task_type': 'CHANGES_DETECTOR_ACCEPTANCE',
            'taskbox_enabled': True,
        }

        deploy_binary_task_parameters = dict(
            arcadia_url=self.Parameters.arcadia_url,
            target='sandbox/projects/devtools/ChangesDetectorAcceptance/bin',
            arcadia_patch=self.Parameters.arcadia_patch,
            attrs=attrs,
            integrational_check=not self.Parameters.just_build,
            integrational_check_payload=[{
                'type': "CHANGES_DETECTOR_ACCEPTANCE",
                'custom_fields': process_custom_fields(integration_payload),
            } for integration_payload in integration_payloads],
            use_arc=True,
            arc_oauth_token=str(self.Parameters.ya_arc_secret),
            resource_for_parent=self.Parameters.release_run,
            binary_executor_release_type="custom",
            taskbox_enabled="on",
        )

        req = deploy_binary_task_parameters.get('__requirements__', {})
        req['tasks_resource'] = self.Requirements.tasks_resource
        deploy_binary_task_parameters.update(__requirements__=req)

        if not self.Parameters.release_run:
            attrs['released'] = 'cancelled'
            deploy_binary_task_parameters['release_ttl'] = 7
        else:
            description = f"[release]\n{description}"

        task_dbt = dbt.DeployBinaryTask(
            self,
            description=description,
            **deploy_binary_task_parameters,
        )
        task_dbt.enqueue()
        self.logger.debug("DeployBinaryTask: %s", task_dbt.id)
        return task_dbt.id

    # CHANGES_DETECTOR checks
    def _do_check_changes_detector(self):
        with self.memoize_stage.find_create_tasks(commit_on_entrance=False):
            task_info = self._search_task()

            self.logger.debug("%s", task_info)

            self._create_tasks(task_info)
        stable_cd = self.server.task[self.Context.cd_task_id].read()
        self.logger.debug("Stable CHANGES_DETECTOR data: %s", stable_cd)
        if stable_cd['status'] != "SUCCESS":
            raise ValueError("Wrong status for original-like task {}".format(stable_cd['id']))
        dbt = self.server.task[self.Context.dbt_task_id].read()
        self.logger.debug("DEPLOY_BINARY_TASK data: %s", dbt)
        if dbt['status'] != "SUCCESS":
            raise ValueError("Wrong status for deploy task {}".format(dbt['id']))
        # assert dbt['children']['count'] == 1
        # self.server.task do not give this information
        new_cd = self.server.task[self.Context.dbt_task_id].children.read()['items'][0]
        self.logger.debug("New CHANGES_DETECTOR data: %s", new_cd)
        if new_cd['status'] != "SUCCESS":
            raise ValueError("Wrong status for testing original-like task {}".format(stable_cd['id']))
        if self.Parameters.precommit_run:
            self._compare_all_graphs(stable_cd, new_cd)
        else:
            self._compare_affected_targets(stable_cd, new_cd)

    def _search_task(self):
        if self.Parameters.precommit_run:
            self.logger.info("Search last stable SUCCESS task")
            task_info = self.server.task.read(
                type="CHANGES_DETECTOR", status="SUCCESS", owner="CI_GRAPH_DISCOVERY", order="-id",
                input_parameters={
                    "binary_executor_release_type": "stable",
                    "build_profile": "pre_commit",
                }, limit=1)["items"][0]
        else:
            self.logger.info("Search last stable SUCCESS task with big affected_targets resource size")
            now = datetime.datetime.utcnow()
            since = now - datetime.timedelta(hours=12)
            self.logger.debug("From %s to %s", since, now)

            resource_info = self.server.resource.read(
                type="AFFECTED_TARGETS", status="SUCCESS", owner="CI_GRAPH_DISCOVERY", order="-size",
                created="{}..{}".format(since.isoformat(), now.isoformat()),
                limit=1
            )["items"][0]

            self.logger.debug("Found resource %d", resource_info['id'])

            task_info = self.server.task[
                resource_info['task']['id']
            ].read()

        self.logger.info("Found task %d", task_info['id'])

        return task_info

    def _create_tasks(self, task_info):
        self.logger.info("Build and run acceptance")
        description = "[GGaaS auto acceptance]\n" \
                      "Check ChangesDetector graph generation"

        integration_payload = {
            **task_info['input_parameters'],

            'compress_results': False,

            'ya_token': str(self.Parameters.ya_arc_secret),
            'binary_executor_release_type': 'custom',
            'push_tasks_resource': False,
        }

        if self.Parameters.precommit_run:
            integration_payload.update({
                'do_not_execute_metagraph': True,

                'affected_targets': False,
                'configure_errors': False,
                'meta_graphs': True,
            })
        else:
            integration_payload.update({
                'do_not_execute_metagraph': False,

                'affected_targets': True,
                'configure_errors': False,
                'meta_graphs': False,
            })

        resource_attrs = {
            'released': 'prestable',
            'task_type': 'CHANGES_DETECTOR',
            'taskbox_enabled': True,
        }

        deploy_binary_task_parameters = dict(
            arcadia_url=self.Parameters.arcadia_url,
            target='sandbox/projects/devtools/ChangesDetector/bin',
            arcadia_patch=self.Parameters.arcadia_patch,
            attrs=resource_attrs,
            integrational_check=not self.Parameters.just_build,
            integrational_check_payload={
                'type': "CHANGES_DETECTOR",
                'custom_fields': process_custom_fields(integration_payload),
            },
            use_arc=True,
            arc_oauth_token=str(self.Parameters.ya_arc_secret),
            resource_for_parent=self.Parameters.release_run,
            binary_executor_release_type="custom",
            taskbox_enabled="on",
        )

        req = deploy_binary_task_parameters.get('__requirements__', {})
        req['tasks_resource'] = self.Requirements.tasks_resource
        deploy_binary_task_parameters.update(__requirements__=req)

        if self.Parameters.precommit_run:
            description = f"[precommit]\n{description}"
        else:
            description = f"[postcommit]\n{description}"

        if not self.Parameters.release_run:
            resource_attrs['released'] = 'cancelled'
            deploy_binary_task_parameters['release_ttl'] = 7
        else:
            description = f"[release]\n{description}"

        task_dbt = dbt.DeployBinaryTask(
            self,
            description=description,
            **deploy_binary_task_parameters,
        )
        task_dbt.enqueue()
        self.Context.dbt_task_id = task_dbt.id
        self.logger.debug("DeployBinaryTask: {}".format(task_dbt.id))

        self.logger.info("Run acceptance task over stable")

        stable_parameters = {
            **integration_payload,
            'binary_executor_release_type': 'stable',
        }

        task_cd = cd.ChangesDetector(
            self,
            description=f"[GGaaS auto acceptance child]\nOriginal task:\n{task_info['id']}:\n{description}",
            **stable_parameters,
        )
        task_cd.enqueue()
        self.Context.cd_task_id = task_cd.id

        raise sdk2.WaitTask(
            [self.Context.dbt_task_id, self.Context.cd_task_id],
            (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
            wait_all=True,
        )

    def _compare_all_graphs(self, control_task, test_task, deep=0):
        self.logger.info("Comparing graphs from %d and %d",
                         control_task['id'], test_task['id'])
        control_graphs = dict(self._get_graphs_from_task(control_task['id'], deep=deep))
        test_graphs = dict(self._get_graphs_from_task(test_task['id'], deep=deep))

        self.logger.info("Found %d graphs from control, %d from test", len(control_graphs), len(test_graphs))

        self.logger.debug("Control graphs: %s", control_graphs.keys())
        self.logger.debug("Test graphs: %s", test_graphs.keys())

        assert len(control_graphs)
        assert len(test_graphs)
        expected = len(set(control_graphs.keys()) & set(test_graphs.keys()))
        assert expected

        correct = 0
        failed = 0

        in_control_not_in_test = set(control_graphs.keys()) - set(test_graphs.keys())
        in_test_not_in_control = set(test_graphs.keys()) - set(control_graphs.keys())

        if in_control_not_in_test:
            msg = "{} graph in control but NOT in test".format(len(in_control_not_in_test))
            self.logger.warning(msg)
            self.set_info(msg)
            self.logger.debug("Graphs: %s ",in_control_not_in_test)
        if in_test_not_in_control:
            msg = "{} graph in test but NOT in control".format(len(in_test_not_in_control))
            self.logger.warning(msg)
            self.set_info(msg)
            self.logger.debug("Graphs: %s", in_test_not_in_control)

        for key, control_graphs_resource in control_graphs.items():
            self.logger.info("============ COMPARE GRAPHS: %s %s/%s ============", *key)

            if key not in test_graphs:
                self.logger.warning(f"Key {key} not found in TEST")
                continue

            test_graphs_resource = test_graphs[key]

            try:
                self._compare_graphs_from(key, control_graphs_resource['id'], test_graphs_resource['id'])
                correct += 1
            except ValueError:
                self.logger.exception("While comparing graphs")
                failed += 1
            except Exception:
                self.logger.exception("Unhandled error while comparing graphs")
                raise

        self.logger.info("Graph comparing complete! %d/%d correct (%d failed)",
                         correct, expected, failed)

        if failed:
            raise ValueError("Found not equal graphs")

        if correct != expected:
            raise ValueError("Too few correct graphs ({}/{}, expect {})".format(
                correct, expected, expected,
            ))

    _key_attributes = ('left_revision',
                       'left_url',
                       'left_patch',
                       'right_revision',
                       'right_url',
                       'right_patch',
                       'circuit_type',
                       'targets',
                       'autocheck_config_path',
                       'partition_index',
                       'partitions_count',
                       )

    def _get_graphs_from_task(self, task_id: int, deep: int = 0):
        self.logger.info("Search meta graph for %d", task_id)
        original_task_id = task_id
        tasks_id = [(task_id, 0)]

        while tasks_id:
            task_id, level = tasks_id.pop()
            task = self.server.task[task_id]
            if level < deep:
                self.logger.debug("Get subtasks from %d (%d)", task_id, level)
                for children_data in task.children.read()['items']:
                    tasks_id.append((children_data['id'], level + 1))

            for item in task.resources.read()['items']:
                if item['type'] == "YA_MAKE_META_GRAPH_RESULT":
                    self.logger.debug("Found %s resource for task %d: %d",
                                      item['type'], task_id, item['id'])
                    key = []
                    use_svn = None
                    for attr_name in self._key_attributes:
                        value = item['attributes'].get(attr_name, None)
                        if value == 'None' or not value:
                            # TODO: Fix in ChangesDetector
                            value = None
                        if attr_name.endswith('_revision'):
                            use_svn = value is not None and int(value) != -1
                            if not use_svn:
                                value = "DISABLED_BY_ARC_HASH"
                        if attr_name.endswith('_url'):
                            if use_svn:
                                value = "DISABLED_BY_SVN_REVISION"

                        key.append(value)
                    yield tuple(key), item
                else:
                    self.logger.debug("Resource type %s, skip", item['type'])

    def _compare_graphs_from(self, key, control_graph_id, test_graph_id):
        self.logger.info("Stable graph resource id: %d", control_graph_id)
        control_resource = sdk2.Resource[control_graph_id]
        control_resource_data = sdk2.ResourceData(control_resource)
        control_graph_path = control_resource_data.path
        self.logger.debug("Path for stable resource: %s", control_graph_path)
        self.logger.info("Source/kind: %s / %s", getattr(control_resource_data, "source", None), getattr(control_resource_data, "kind", None))

        self.logger.info("New graph resource id: %d", test_graph_id)
        test_resource = sdk2.Resource[test_graph_id]
        test_resource_data = sdk2.ResourceData(test_resource)
        test_graph_path = test_resource_data.path
        self.logger.debug("Path for new resource: %s", test_graph_path)
        self.logger.info("Source/kind: %s / %s", getattr(test_resource_data, "source", None), getattr(test_resource_data, "kind", None))

        for side in ('left', 'right', 'subtract'):
            self.logger.info("~~~~ Compare %s ~~~~~~~", side)

            with (control_graph_path / f"{side}.json").open("rt") as f:
                stable_json = json.load(f)

            with (test_graph_path / f"{side}.json").open("rt") as f:
                new_json = json.load(f)

            logger = logging.getLogger("versus")

            graph_difference = []

            for _ in Graph.versus_dict(stable_json, new_json):
                logger.info(str(_))
                graph_difference.append(str(_))

            stable_id = None
            new_id = None

            if side == 'subtract':
                stable_id = stable_json['graph']['result'][0]
                new_id = new_json['graph']['result'][0]
            else:
                stable_id = stable_json['result'][0]
                new_id = new_json['result'][0]

            # TODO: Store differences

            common_info = (
                "Resource key: {}<br>"
                "Side: {}<br>"
                "Control resource: {}<br>"
                "Test resource: {}<br>".format(
                    key,
                    side,
                    control_resource.url,
                    test_resource.url,
                ))

            if stable_id == new_id:
                self.set_info(
                    "{}"
                    "<br>Graphs id is equal: <code>{}</code><br>"
                    "Also we have {} differences (<i>not different items!</i>) in graph,"
                    " but don't care about it. <i>But if you care, check it in log</i>".format(
                        common_info,
                        new_id, len(graph_difference)
                    ), do_escape=False)
            else:
                self.set_info(
                    "{}"
                    "<br><font color=red>Have a different graph id</font>: <code>{}</code> in control vs <code>{}</code> in test<br>"
                    "<i>Not all differences affect changing graph id!</i><br>"
                    "Here is differences (control vs test, {}):<br>"
                    "{}{}".format(
                        common_info,
                        stable_id, new_id,
                        len(graph_difference),
                        "<br>".join(("<code>{}</code>".format(item) for item in graph_difference[:10])),
                        "<br>..." if len(graph_difference) > 10 else "",  # TODO: Make this link
                ), do_escape=False)

                raise ValueError("Different graphs id: {} vs {}".format(
                    stable_id, new_id
                ))

    def _compare_affected_targets(self, stable_cd, new_cd):
        self.logger.info("Comparing affected_targets from %d and %d",
                         stable_cd['id'], new_cd['id'])

        stable_resource = self._get_affected_targets_from_task(stable_cd['id'])
        new_resource = self._get_affected_targets_from_task(new_cd['id'])

        assert stable_resource
        assert new_resource

        stable_data = sdk2.ResourceData(sdk2.Resource[stable_resource['id']]).path
        self.logger.debug("Path for stable resource: %s", stable_data)

        new_data = sdk2.ResourceData(sdk2.Resource[new_resource['id']]).path
        self.logger.debug("Path for new resource: %s", new_data)

        with stable_data.open("rt") as f:
            stable_json = json.load(f)

        with new_data.open("rt") as f:
            new_json = json.load(f)

        in_stable_not_new = set(stable_json) - set(new_json)
        if in_stable_not_new:
            self.logger.warning("Targets in stable, not in new: %s", in_stable_not_new)
            self.set_info("Targets in stable, not in new: {}".format(in_stable_not_new), do_escape=False)
        in_new_not_stable = set(new_json) - set(stable_json)
        if in_new_not_stable:
            self.logger.warning('Targets in new, not in stable: %s', in_new_not_stable)
            self.set_info("Targets in new, not in stable: {}".format(in_new_not_stable), do_escape=False)

        if set(stable_json) != set(new_json):
            raise ValueError("Different affected targets ({} differences)".format(
                len(in_stable_not_new) + len(in_new_not_stable)
            ))

    def _get_affected_targets_from_task(self, changes_detector_id: int):
        self.logger.info("Search affected_targets for %d", changes_detector_id)
        for item in self.server.task[changes_detector_id].resources.read()['items']:
            if item['type'] == "AFFECTED_TARGETS":
                self.logger.debug("Found %s resource for task %d: %d",
                                  item['type'], changes_detector_id, item['id'])
                return item
            else:
                self.logger.debug("Resource type %s, skip", item['type'])
