import logging
import datetime
import requests

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.types import client as ctc


class CoverageDelay(sdk2.Task):
    SOLOMON_API_URL = 'http://api.solomon.search.yandex.net'
    ARCANUM_API_URL = "https://a.yandex-team.ru/api"

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT

    class Parameters(sdk2.Parameters):
        arcanum_token_vault_name = sdk2.parameters.String(
            'Arcanum token secret name',
            default='arcadia-devtools-arcanum-token',
            required=True,
        )
        solomon_token_vault_name = sdk2.parameters.String(
            'Solomon token secret name',
            default='arcadia-devtools-solomon-token',
            required=True,
        )

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"released": ctt.ReleaseStatus.STABLE, "task_type": "CoverageDelay"}
        ).first()

    def on_execute(self):
        current = requests.get(
            url="{ARCANUM_API}/tree/node/trunk/arcadia/?repo=arc".format(ARCANUM_API=self.ARCANUM_API_URL),
            headers={'Authorization': 'OAuth %s' % sdk2.Vault.data(self.owner, self.Parameters.arcanum_token_vault_name)},
            verify=False
        ).json()
        logging.info("Current coverage: %s", current["coverage"])
        try:
            lastUpdate = datetime.datetime.strptime(current["coverage"]["lastUpdate"], "%Y-%m-%dT%H:%M:%S.%fZ")
        except ValueError:
            logging.exception("Cannot parse lastUpdate = {}".format(current["lastUpdate"]))
            raise
        utcnow = datetime.datetime.utcnow()
        total_seconds_diff = int((utcnow - lastUpdate).total_seconds())
        self.set_info("Delay: {}".format(total_seconds_diff))
        self._send_to_solomon(
            self.SOLOMON_API_URL, sdk2.Vault.data(self.owner, self.Parameters.solomon_token_vault_name),
            "coverage", "arcadia", "nightly", {"delay_seconds":  total_seconds_diff, "delay_hours": total_seconds_diff / 3600},
        )

    def _send_to_solomon(self, api_url, api_token, project, cluster_name, service, metrics):
        from solomon import PushApiReporter, OAuthProvider
        pusher = PushApiReporter(
            project=project,
            cluster=cluster_name,
            service=service,
            url=api_url,
            auth_provider=OAuthProvider(api_token)
        )
        for key, value in metrics.iteritems():
            pusher.set_value(key, value)
