# coding: utf-8

import collections
import json
import logging
import os
import re
import signal
import threading
import time
import urllib
import uuid

import psutil

import sandbox.common
import sandbox.sandboxsdk as ssdk

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import constants as const
from sandbox.projects.common import solomon
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build import YaMake2 as ya_make_2
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params
from sandbox.projects import resource_types


def run_watch_dog(deadline):
    assert deadline > 0, deadline
    ts = time.time()
    while deadline > ts:
        ts = time.time()
        sleep = ((deadline - ts) // 2) + 1
        logging.debug("watch dog is going to sleep for %d secs", sleep)
        time.sleep(sleep)

    logging.debug("Task deadline time exceeded (deadline:{})".format(deadline))
    try:
        for proc in psutil.Process(os.getpid()).children():
            try:
                os.killpg(proc.pid, signal.SIGTERM)
            except OSError as e:
                logging.error("Failed to killpg %d: %s", proc.pid, e)
        time.sleep(5)
    except Exception as e:
        logging.debug("Error: %s", e)

    os.kill(os.getpid(), signal.SIGTERM)


class YtTokenVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'yt_token_vault_key'
    description = 'YT token vault key to upload coverage data'
    required = True


class YtexecTokenVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'ytexec_token_vault_key'
    description = 'YT token vault key to run tests over YT using ytexec'
    required = True


class TestSizeFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = const.TEST_SIZE_FILTER
    description = 'Test size filter'
    default_value = 'small,medium,large'


class DownloadArtifactsParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.DOWNLOAD_ARTIFACTS_FROM_DISTBUILD
    description = 'Download build artifacts when using distributed build'
    default_value = True


class CheckYaReturnCodeParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'check_ya_return_code'
    description = 'Check ya rc is 0 or 10'
    default_value = True


class CustomBuildFlags(ssdk.parameters.SandboxStringParameter):
    name = 'custom_build_flags'
    description = 'Custom build flags (space-separated without "-D" prefix)'
    default_value = ""


class TestTypesFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = 'test_types_filter'
    description = 'Test types (coma-separated)'
    default_value = 'unittest,coverage_extractor,pytest,gtest,boost_test,java,exectest'


class TestTagsFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = const.TEST_TAG
    description = 'Test tags'
    default_value = None


class DistBuildLowPriorityParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'low_dist_priority'
    description = 'Decrease dist priority to avoid crowding out work tasks'
    default_value = False


class DistBuildCoordinatorsFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = const.COORDINATORS_FILTER
    description = 'Distbuild coordinators filter'
    default_value = 'distbuild-man'


class SaveLinksForFileParameter(ssdk.parameters.SandboxStringParameter):
    name = const.SAVE_LINKS_FOR_FILES
    description = 'Save links instead of files for specified filename (coma-separated)'
    # don't bring test's output data - it's not really interesting, don't bring coverage.tar it's too huge
    default_value = ','.join([
        'coverage.tar',
        'go.coverage.tar',
        'java.coverage.tar',
        'nlg.coverage.tar',
        'py2.coverage.tar',
        'py3.coverage.tar',
        'unified.coverage.tar',
        'testing_out_stuff.tar',
        'testing_out_stuff.tar.zstd',
        # necessary optimization - ya runner takes to much time to download artifacts,
        # skip some extra output files (95% of all output data size)
        'clangcov_resolve.*.log',
        'clangcov_resolve.done',
        'coverage_merge.log',
        'results_accumulator.log',
        'coverage_merge_res_accumulator.log',
        'coverage_resolved.*_upload.log',
        'coverage_resolved.*.json',
        '*_coverage_resolve.log',
        'javacov_resolve.log',
        'run_test.log',
    ])


class ContainerParameter(ssdk.parameters.Container):
    name = const.SANDBOX_CONTAINER
    description = 'Container the task should execute in'
    default_value = None
    required = False


class FastClangCoverageMergeParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.FAST_CLANG_COVERAGE_MERGE
    description = 'Use fuse to accumulate and merge coverage in memory'
    default_value = True


class BuildSystem(ssdk.parameters.SandboxStringParameter):
    name = const.BUILD_SYSTEM_KEY
    description = 'Build system'
    required = True
    default_value = const.DISTBUILD_FORCE_BUILD_SYSTEM
    choices = [
        ('ya force', const.YA_MAKE_FORCE_BUILD_SYSTEM),
        ('semi-distbuild', const.SEMI_DISTBUILD_BUILD_SYSTEM),
        ('distbuild force', const.DISTBUILD_FORCE_BUILD_SYSTEM),
    ]


class TaskDeadlineTimeStampParameter(ssdk.parameters.SandboxStringParameter):
    name = 'task_deadline_timestamp'
    description = 'Task deadline timestamp'
    default_value = 0


class UploadCoverageParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.UPLOAD_COVERAGE
    description = 'Upload coverage'
    default_value = True


class PushStatsToSolomonParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'push_stats_to_solomon'
    description = 'Push stats to solomon'
    default_value = False


class SolomonCommonLabelsParameter(ssdk.parameters.SandboxStringParameter):
    name = 'solomon_common_labels'
    description = 'Solomon common labels'
    default_value = ''


class MergeCoverageParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.MERGE_COVERAGE
    description = 'Merge coverage'
    default_value = False


class ProjectsPartitionsCount(ssdk.parameters.SandboxStringParameter):
    name = 'projects_partitions_count'
    description = 'Projects partitions count'
    default_value = 1


class ProjectsPartitionIndex(ssdk.parameters.SandboxStringParameter):
    name = 'projects_partition_index'
    description = 'Projects partition index'
    default_value = 0


class BuildExecutionTime(ssdk.parameters.SandboxIntegerParameter):
    name = const.BUILD_EXECUTION_TIME
    description = 'Build execution time'
    default_value = 2 * 3600  # 2h


class CoverageYaMakeTask(YaMakeTask):

    type = 'COVERAGE_YA_MAKE_TASK'
    client_tags = ya_make_2.YA_MAKE_CLIENT_TAGS & sandbox.common.types.client.Tag.HDD

    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        build_params.VaultOwner,
        build_params.EnvironmentVarsParam,
        build_params.DirOutputs,
        TestSizeFilterParameter,
        TestTypesFilterParameter,
        TestTagsFilterParameter,
        YtTokenVaultKeyName,
        YtexecTokenVaultKeyName,
        DownloadArtifactsParameter,
        CheckYaReturnCodeParameter,
        CustomBuildFlags,
        DistBuildLowPriorityParameter,
        DistBuildCoordinatorsFilterParameter,
        SaveLinksForFileParameter,
        ContainerParameter,
        FastClangCoverageMergeParameter,
        BuildSystem,
        TaskDeadlineTimeStampParameter,
        BuildExecutionTime,
        UploadCoverageParameter,
        MergeCoverageParameter,
        ProjectsPartitionsCount,
        ProjectsPartitionIndex,
        PushStatsToSolomonParameter,
        SolomonCommonLabelsParameter,
    ] + get_arcadia_project_base_target_params().params

    def get_build_def_flags(self):
        build_def_flags = YaMakeTask.get_build_def_flags(self) or ""
        projects_partitions_count = int(self.ctx.get(ProjectsPartitionsCount.name, ProjectsPartitionsCount.default_value))
        if projects_partitions_count > 1:
            build_def_flags += " -DRECURSE_PARTITIONS_COUNT={} -DRECURSE_PARTITION_INDEX={}".format(
                projects_partitions_count,
                self.ctx.get(ProjectsPartitionIndex.name, ProjectsPartitionIndex.default_value)
            )
        return build_def_flags

    def pre_execute(self):
        YaMakeTask.pre_execute(self)
        logging.info("pre_execute")

        self.setup_deadline_timer()

        self.stats = collections.defaultdict(int)
        # setup build and run options
        if const.BUILD_SYSTEM_KEY not in self.ctx:
            self.ctx[const.BUILD_SYSTEM_KEY] = const.DISTBUILD_FORCE_BUILD_SYSTEM
        # doesn't work properly - https://paste.yandex-team.ru/392932/text
        # self.ctx[const.MAKE_CONTEXT_ON_DISTBUILD] = True
        # remove after fix with MAKE_CONTEXT_ON_DISTBUILD
        self.ctx[const.SANDBOX_TAGS] = ''
        if self.ctx.get('low_dist_priority'):
            self.ctx[const.DISTBUILD_PRIORITY] = '-300000000'
        self.ctx[const.NEW_DIST_MODE] = True
        self.ctx[const.KEEP_ON] = True
        self.ctx[const.SKIP_TEST_CONSOLE_REPORT] = True
        self.ctx[const.DROP_GRAPH_RESULT_BEFORE_TESTS] = True
        self.ctx[const.CREATE_RESULTS_RESOURCE] = False
        self.ctx[const.CHECK_RETURN_CODE] = False
        self.ctx[const.STRIP_SKIPPED_TEST_DEPS] = True
        self.ctx[const.GRAPH_TIMESTAMP] = self.ctx.get(const.GRAPH_TIMESTAMP, int(time.time()))
        # Some extra time for ya-make and sandbox machinery
        self.ctx[const.YA_TIMEOUT] = self.ctx.get(const.YA_TIMEOUT, self.ctx[const.BUILD_EXECUTION_TIME] + 3600)
        custom_build_flags = filter(None, [s.strip() for s in self.ctx.get('custom_build_flags').split(' ')])

        if self.ctx.get(const.COVERAGE_UNIFIED_AGENT):
            logging.info('New upload scheme will be used')
            if not self.ctx.get(const.COVERAGE_UNIFIED_AGENT_SID):
                sid = str(uuid.uuid4())
                self.ctx[const.COVERAGE_UNIFIED_AGENT_SID] = sid
                logging.warning('Sid is not specified, {} will be used'.format(sid))

        # XXX temporary fix - see https://st.yandex-team.ru/TESTENV-2520
        for name in [const.STREAMING_REPORT_URL, const.STREAMING_REPORT_ID]:
            if name in self.ctx:
                del self.ctx[name]

        if custom_build_flags:
            logging.warning("Test and build results won't be uploaded to CI due custom build flags are used witch will lead to changed toolchain name. "
                            "CI is waiting only records with specified in autocheck-config-coverage.json toolchains, in order to avoid spam emails, "
                            "because coverage runs once a night and fixes with failures are irrelevant to the commit.")
            # disable streaming
            for name in [const.STREAMING_REPORT_URL, const.STREAMING_REPORT_ID]:
                if name in self.ctx:
                    del self.ctx[const.STREAMING_REPORT_URL]
        else:
            self.ctx[const.REPORT_CONFIG_PATH] = "autocheck/autocheck-config-coverage.json"
        self.ctx[const.DEFINITION_FLAGS_KEY] = ' '.join(['-D{}'.format(v) for v in custom_build_flags])
        # Task doesn't download logs from distbuild, instead it creates files with link path
        # to resolve such logs we should not specify results root
        self.ctx[const.LOCAL_OUTPUT_DIR] = True  # TODO FIX IT - False
        # setup coverage uploading
        self.ctx[const.CPP_COVERAGE_TYPE] = 'clang'
        self.ctx[const.PYTHON_COVERAGE] = True
        self.ctx[const.JAVA_COVERAGE] = True
        self.ctx[const.GO_COVERAGE] = True
        self.ctx[const.NLG_COVERAGE] = True
        self.ctx[const.RUN_TAGGED_TESTS_ON_YT] = True
        self.ctx[const.UPLOAD_COVERAGE] = self.ctx.get(const.UPLOAD_COVERAGE, True)
        self.ctx[const.MERGE_COVERAGE] = self.ctx.get(const.MERGE_COVERAGE, False)
        # always set yt token path - it will be used in the coverage_uploader nodes in the local-mode
        if self.ctx[const.UPLOAD_COVERAGE]:
            yt_token_path = os.path.abspath("secret")
            fu.write_file(yt_token_path, self.get_vault_data(self.ctx.get(const.VAULT_OWNER, self.author), self.ctx.get('yt_token_vault_key')))
            # Setup YT token for coverage uploading nodes
            self.ctx[const.COVERAGE_YT_TOKEN_PATH] = yt_token_path
        else:
            self.ctx[const.COVERAGE_YT_TOKEN_PATH] = ""
        # Setup YT token for ytexec test nodes
        if self.ctx.get(YtexecTokenVaultKeyName.name):
            # Use common machinery to obtain token https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/build/YaMake/__init__.py?rev=r8266627#L557
            self.ctx[const.YA_YT_TOKEN_VAULT_OWNER] = self.ctx.get(const.VAULT_OWNER, self.author)
            self.ctx[const.YA_YT_TOKEN_VAULT_NAME] = self.ctx.get(YtexecTokenVaultKeyName.name)
        # setup tests
        self.ctx[const.TESTS_REQUESTED] = True
        self.ctx[const.MERGE_SPLIT_TESTS] = False
        self.ctx[const.TEST_TAG] = self.ctx.get(const.TEST_TAG, '-ya:force_sandbox')
        self.ctx[const.TEST_TYPE_FILTER] = self.ctx.get('test_types_filter')
        if self.ctx.get(const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH, ''):
            self.ctx[const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH] = os.path.join(os.getcwd(), self.ctx[const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH])
            self.ctx[const.COVERAGE_UNIFIED_AGENT_FAILED_UIDS_FILE_PATH] = self.ctx[const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH] + '.nodes.failed'
        self.start_timestamp = time.time()

    def setup_deadline_timer(self):
        deadline_timestamp = int(self.ctx.get('task_deadline_timestamp') or 0)
        logging.debug("deadline_timestamp:%d time:%s", deadline_timestamp, time.time())

        if not deadline_timestamp:
            return

        if time.time() > deadline_timestamp:
            raise sandbox.common.errors.TaskFailure("Deadline time exceeded")

        thread = threading.Thread(target=run_watch_dog, args=(deadline_timestamp,))
        thread.daemon = True
        thread.start()

    def post_build(self, source_dir, output_dir, pack_dir):
        self.stats['ya-bin walltime'] = time.time() - self.start_timestamp

        if self.ctx.get('check_ya_return_code') and self.ctx.get('build_returncode') not in [0, 10]:
            raise sandbox.common.errors.TemporaryError("ya make failed with {} returncode".format(self.ctx.get('build_returncode')))

        self.results_filename = os.path.join(output_dir, 'results.json')
        if os.path.exists(self.results_filename):
            self.parse_tests_info(output_dir)
        else:
            logging.warning("Looks like infrastructure error had occurred. Don't publish partly collected coverage to the CI and Arcanum to avoid irrelevant peaks on the charts")
            raise sandbox.common.errors.TemporaryError("Looks like infrastructure error had occurred")

        self.check_clang_coverage_merge_logs(output_dir)

        if self.ctx.get(PushStatsToSolomonParameter.name):
            self.push_stats_to_solomon()
        self.set_description()

        if not self.stats.get('test projects completely processed'):
            raise sandbox.common.errors.TemporaryError("Zero projects were successfully processed")

        processed = self.stats.get('test projects completely processed (%)', 0.0)
        # XXX https://ml.yandex-team.ru/thread/devtools/166914661189480185/
        if processed < 80.0:
            raise sandbox.common.errors.TemporaryError("Too few projects ({}) have been correctly processed - looks like infra problem".format(processed))

        if (
            self.ctx.get(const.UPLOAD_COVERAGE, False) and
            self.ctx.get(const.COVERAGE_UNIFIED_AGENT, False) and
            self.ctx.get(const.COVERAGE_UNIFIED_AGENT_UIDS_RESOURCE_ID, 0) and
            self.ctx.get(const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH, '')
        ):
            coverage_unified_agent_uids_file_path = self.ctx.get(const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH, '')
            coverage_unified_agent_uids_res_id = self.ctx.get(const.COVERAGE_UNIFIED_AGENT_UIDS_RESOURCE_ID, 0)
            if not os.path.exists(coverage_unified_agent_uids_file_path):
                logging.warning(
                    "File with uploaded nodes uids is not existed (%s file expected)",
                    coverage_unified_agent_uids_file_path
                )
                with open(coverage_unified_agent_uids_file_path, 'w'):
                    pass

            ssdk.channel.channel.task.save_parent_task_resource(
                os.path.dirname(coverage_unified_agent_uids_file_path), coverage_unified_agent_uids_res_id
            )
            ssdk.channel.channel.task.mark_resource_ready(coverage_unified_agent_uids_res_id)

            coverage_unified_agent_failed_uids_file_path = self.ctx.get(const.COVERAGE_UNIFIED_AGENT_FAILED_UIDS_FILE_PATH, '')
            if os.path.exists(coverage_unified_agent_failed_uids_file_path):
                res = self.create_resource("Failed coverage upload uids file", os.path.dirname(coverage_unified_agent_failed_uids_file_path), resource_types.OTHER_RESOURCE)
                self.mark_resource_ready(res)

    def set_description(self):
        self.set_info(json.dumps(self.stats, sort_keys=True, indent=4))

    def push_stats_to_solomon(self):
        if not self.ctx.get(SolomonCommonLabelsParameter.name):
            logging.error("Failed to find %s", SolomonCommonLabelsParameter.name)
            return

        try:
            common_labels = json.loads(self.ctx[SolomonCommonLabelsParameter.name])

            token = self.get_vault_data('AUTOCHECK', 'devtools-ya-coverage-solomon-token')
            timestamp = int(self.ctx[const.GRAPH_TIMESTAMP])
            sensors = [
                {
                    'labels': {'sensor': k},
                    'ts': timestamp,
                    'value': v,
                }
                for k, v in self.stats.items()
            ]
            params = {
                'project': 'coverage',
                'cluster': 'arcadia',
                'service': 'nightly',
            }

            solomon.push_to_solomon_v2(token=token, params=params, common_labels=common_labels, sensors=sensors)
        except Exception as e:
            logging.exception('Failed to push stats to solomon: %s', e)

    def parse_tests_info(self, output_dir):
        coverage_chains = {
            'clangcov_resolve.done': [
                # TODO check upload output nodes after changes in @3916089
            ],
            'python_coverage_resolve.log': [
                'coverage_resolved.python.json',
            ],
            'go_coverage_resolve.log': [
                'coverage_resolved.go.json',
            ],
            'javacov_resolve.log': [
                'coverage_resolved.java.json',
            ],
        }

        if self.ctx[const.UPLOAD_COVERAGE]:
            coverage_chains['python_coverage_resolve.log'].append('coverage_resolved.python_upload.log')
            coverage_chains['go_coverage_resolve.log'].append('coverage_resolved.go_upload.log')
            coverage_chains['javacov_resolve.log'].append('coverage_resolved.java_upload.log')

        logging.info("Parsing results.json")
        with open(self.results_filename) as afile:
            data = json.load(afile)['results']

        chunks = collections.defaultdict(list)
        suites = []
        for e in data:
            if e.get('chunk'):
                chunks[e['path']].append(e)
            if e.get('suite'):
                suites.append(e)

        def report_problem(project, msg):
            self.stats[msg] += 1
            logging.warning("%s: %s", project, msg)

        def file_presented(testdir, filename, report):
            absname = os.path.join(testdir, filename)
            if not os.path.exists(absname) and not os.path.exists(absname + '.link'):
                if report:
                    self.stats['test projects without {}'.format(filename)] += 1
                    logging.warning("%s: test project without %s (%s)", e.get('path'), filename, absname)
                return False
            return True

        seen = set()
        relpath_regexp = re.compile(r'proxy\.sandbox\.yandex-team\.ru/\d+/(.*?/test-results/\S+?)/')

        for e in sorted(suites, key=lambda x: x.get('path')):
            self.stats['test projects total'] += 1
            if e.get('error_type'):
                self.stats['test projects {}'.format(e.get('error_type'))] += 1

                if e.get('error_type') == 'BROKEN_DEPS':
                    logging.debug("%s: test project with broken depends", e.get('path'))
                    # there are links provided for broken targets
                    continue

            if e.get('path') not in chunks:
                self.stats['test projects without chunks'] += 1
                logging.debug("%s: test projects without chunks", e.get('path'))
                continue

            for chunk in chunks[e.get('path')]:
                links = chunk.get('links')
                if not links:
                    report_problem(chunk.get('path'), 'test chunks without links')
                    continue

                logsdir = links.get('logsdir')
                if not logsdir:  # no forks
                    report_problem(e.get('path'), 'test chunk without logsdir')
                    logging.warning("chunk info %s", chunk)
                    continue

                match = relpath_regexp.search(logsdir[0])
                assert match, (logsdir[0])
                relpath = urllib.unquote(match.group(1))
                testdir = os.path.join(output_dir, relpath)
                assert os.path.exists(testdir), (testdir, relpath, match.group(1), output_dir)

                if testdir in seen:
                    continue
                else:
                    seen.add(testdir)

                if not file_presented(testdir, 'run_test.log', report=True):
                    continue

                def chain_presented(filenames):
                    for filename in filenames:
                        if not file_presented(testdir, filename, report=True):
                            return False
                    return True

                found_chains = []
                missing_files = False
                for head_file, rest_files in coverage_chains.items():
                    if not file_presented(testdir, head_file, report=False):
                        continue

                    found_chains.append(head_file)
                    if not chain_presented(rest_files):
                        missing_files = True

                if not found_chains:
                    report_problem(chunk.get('path'), 'test projects without any coverage data')
                    continue

                if missing_files:
                    logging.warning("Missing files found")
                    continue

                self.stats['test chunks completely processed'] += 1

            self.stats['test projects completely processed'] += 1

        if self.stats['test projects total']:
            self.stats['test projects completely processed (%)'] = float(self.stats.get('test projects completely processed', 0)) / self.stats['test projects total'] * 100
        else:
            self.stats['test projects completely processed (%)'] = 0

    def check_clang_coverage_merge_logs(self, output_dir):
        merge_log_names = ['coverage_merge.log', 'coverage_merge_res_accumulator.log']
        panic_prefix = '!!PANIC!!: '

        for root, dirs, files in os.walk(output_dir):
            for name in merge_log_names:
                if name not in files:
                    continue

                filename = os.path.join(root, name)
                relname = os.path.relpath(filename, output_dir)

                with open(filename) as afile:
                    for line in afile:
                        if panic_prefix in line:
                            self.stats['coverage merge panics'] += 1
                            logging.warning("Panic found in %s: %s", relname, line)
                            continue


__Task__ = CoverageYaMakeTask
