import os
import sys
import copy
import json
import logging
import time
import subprocess
import uuid
import datetime

import sandbox.sandboxsdk as ssdk

from sandbox.common.types.task import Status, TaskStatus
from sandbox.projects import resource_types
from sandbox.projects.common import constants as const
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import solomon
from sandbox.projects.common.arcadia import sdk_tested_parts as stp
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build.sdk import sdk_compat


COVERAGE_MERGER = 'devtools/ya/test/programs/coverage_merger/bin/coverage_merger'
COVERAGE_SPREADER = 'devtools/ya/test/programs/coverage_spreader/bin/coverage_spreader'
COVERAGE_WAIT_TOOL = 'devtools/ya/test/programs/coverage_wait_tool/coverage_wait_tool'
SRCS_EXTS = ['java', 'py', 'pyx', 'pxi', 'c', 'cpp', 'h', 'cc', 'hh', 'cxx', 'hxx', 'ipp', 'hpp', 'C', 'H', 'go']
TASK_KILL_TIMEOUT = 28800  # 8h
YT_ROOT_PATH = '//home/codecoverage/v1'
UPLOADED_UIDS_FILE_SALTED = 'uids_{}/uids.list'

INITIALIZED = '__initialized'
SUBTASK_TYPE = 'subtask_type'


class YtTokenVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'yt_token_vault_key'
    description = 'YT token vault key to upload coverage data'
    required = True


class YtexecTokenVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'ytexec_token_vault_key'
    description = 'YT token vault key to run tests over YT using ytexec'
    default_value = ''


class PostgresPasswordVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'postgres_password_vault_key'
    description = 'Postgres password to upload coverage data to the Arcanum'
    default_value = ''


class PostgresProdPasswordVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'postgres_prod_password_vault_key'
    description = 'Postgres prod password to upload coverage data to the Arcanum'
    default_value = ''


class PublishCoverageToArcanumParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'publish_coverage_arcanum'
    description = "Upload merged data from YT to the arcanum's test postgres db"
    default_value = True


class PublishCoverageToArcanumProdParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'publish_coverage_arcanum_prod'
    description = "Upload merged data from YT to the arcanum's prod postgres db"
    default_value = False


class UploadDirStatsParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'upload_dir_stat'
    description = 'Upload coverage stats for directory to the Arcanum'
    default_value = True


class FastClangCoverageMergeParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.FAST_CLANG_COVERAGE_MERGE
    description = 'Use fuse to accumulate and merge coverage in memory'
    default_value = True


class ExtraTasksTestTagParameter(ssdk.parameters.SandboxStringParameter):
    name = 'extra_tasks_test_tag'
    description = 'Create subtasks for tests with specified tag'
    default_value = 'ya:sandbox_coverage'


class TestTypesFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = 'test_types_filter'
    description = 'Test types (coma-separated)'
    default_value = 'unittest,coverage_extractor,pytest,gtest,boost_test,java,exectest,go_test'


class TargetForDistbuildTaskParameter(ssdk.parameters.SandboxStringParameter):
    name = 'main_task_target'
    description = 'Main task target'
    default_value = 'autocheck'


class UseUnifiedAgentParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.COVERAGE_UNIFIED_AGENT
    description = 'Use unified agent for coverage uploading'
    default_value = False


class UseUnifiedAgentStrictParameter(ssdk.parameters.SandboxBoolParameter):
    name = const.COVERAGE_UNIFIED_AGENT_STRICT
    description = 'Upload coverage only with unifed agent'
    default_value = False


class DistBuildCoordinatorsFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = const.COORDINATORS_FILTER
    description = 'Distbuild coordinators filter'
    default_value = 'distbuild-man'


class TestSizeFilterParameter(ssdk.parameters.SandboxStringParameter):
    name = const.TEST_SIZE_FILTER
    description = 'Test size filter'
    default_value = 'small,medium'


class PushStatsToSolomonParameter(ssdk.parameters.SandboxBoolParameter):
    name = 'push_stats_to_solomon'
    description = 'Push stats to solomon'
    default_value = False


class DistbuildPool(build_params.DistbuildPool):
    default_value = "//sas/users/devtools/coverage"


class TestInfo(object):
    def __init__(self, dart_info):
        raw_tags = dart_info.get('TAG', [])
        self.project_path = dart_info['SOURCE-FOLDER-PATH']
        self.size = dart_info['SIZE']
        self.tags = '+'.join(raw_tags)
        self.privileged = 'ya:privileged' in raw_tags
        self.requirements = self.get_requirements(dart_info)

    @staticmethod
    def get_requirements(dart_info):
        res = {}
        for entry in dart_info.get('REQUIREMENTS', []):
            if ":" in entry:
                name, val = entry.split(":", 1)
                res[name] = val
        return res

    def get_container_id(self):
        if 'container' not in self.requirements:
            return None

        try:
            return int(self.requirements.get('container'))
        except Exception as e:
            logging.debug("Failed to get container id: %s", e)
            return None

    def use_dns64(self):
        return self.requirements.get('dns') == 'dns64'

    def __str__(self):
        return 'TestInfo([project_path={} requirements={}])'.format(self.project_path, self.requirements)

    def __repr__(self):
        return str(self)


class SyncFuture(object):
    def __init__(self, spawn):
        self._w = spawn()

    def result(self):
        return self._w()


class CoverageYaMakeSeminanti(ssdk.task.SandboxTask):

    type = "COVERAGE_YA_MAKE_TASK_SEMINANTI"
    execution_space = 1024

    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.VaultOwner,
        DistbuildPool,
        YtTokenVaultKeyName,
        YtexecTokenVaultKeyName,
        PostgresPasswordVaultKeyName,
        PostgresProdPasswordVaultKeyName,
        PublishCoverageToArcanumParameter,
        PublishCoverageToArcanumProdParameter,
        UploadDirStatsParameter,
        FastClangCoverageMergeParameter,
        ExtraTasksTestTagParameter,
        TestTypesFilterParameter,
        TestSizeFilterParameter,
        TargetForDistbuildTaskParameter,
        UseUnifiedAgentParameter,
        UseUnifiedAgentStrictParameter,
        PushStatsToSolomonParameter,
    ]

    def on_execute(self):
        if INITIALIZED not in self.ctx:
            self.ctx[INITIALIZED] = True
            self.initialize()
        else:
            self.finalize()

    def initialize(self):
        logging.debug("initialize() called")
        self.setup_params()
        coverage_merger = self.create_build_task(COVERAGE_MERGER)
        coverage_spreader = self.create_build_task(COVERAGE_SPREADER)
        waiters = [coverage_merger, coverage_spreader]
        if self.ctx.get(const.COVERAGE_UNIFIED_AGENT, False):
            waiters.append(self.create_build_task(COVERAGE_WAIT_TOOL))
        # Start distbuild coverage task as soon as possible
        main_tasks = self.create_distbuild_coverage_tasks()
        with self.with_source_dir() as self.source_root:
            # Large tests with 'ya:force_sandbox' tag
            targets = self.obtain_extra_targets()
            subtasks = self.create_large_coverage_subtasks(targets)
            self.wait_subtasks(main_tasks + waiters + subtasks)

    def finalize(self):
        logging.debug("finalize() called")

        with self.with_source_dir() as self.source_root:
            self.setup_finalization()
            wait_tool_exit_code = 0
            if self.ctx.get(const.COVERAGE_UNIFIED_AGENT, False):
                self.setup_unified_agent_finalization()
                wait_tool_exit_code = self.run_wait_tool()
            self.merge_coverage(self.revision, self.snapshot)

            if self.ctx.get(PushStatsToSolomonParameter.name):
                covdata_stats_future = SyncFuture(self.spawn_calc_coverage_stats)
            else:
                covdata_stats_future = SyncFuture(lambda: lambda: {})

            failed_main_tasks = []
            for main_task in self.get_main_subtasks():
                if main_task.status != TaskStatus.FINISHED:
                    failed_main_tasks.append(main_task)

            self.hash_map_proc.terminate()
            # Publish coverage data only if there are no failed main tasks
            if not failed_main_tasks:
                if self.spread_coverage(self.revision, self.snapshot, self.source_root):
                    # Update 'last' link only if spread_coverage succeed
                    self.update_last_link(self.revision, self.snapshot, self.source_root)

            failed_subtasks = [t for t in self.get_extra_subtasks() if t.status != TaskStatus.FINISHED]

            self.stats.update(
                {
                    'failed main tasks': len(failed_main_tasks),
                    'failed subtasks': len(failed_subtasks),
                    'total task wall time': int(time.time()) - int(self.ctx[const.GRAPH_TIMESTAMP]),
                }
            )

            if self.ctx.get(PushStatsToSolomonParameter.name):
                self.stats.update(covdata_stats_future.result().items())
                self.push_stats_to_solomon()
            self.set_description()

            if failed_main_tasks:
                raise ssdk.errors.SandboxTaskFailureError("{} main subtasks failed".format(len(failed_main_tasks)))

            errors = []
            if failed_subtasks:
                errors.append("{} extra subtasks failed".format(len(failed_subtasks)))
            if wait_tool_exit_code:
                errors.append("Not all coverage delivered")
            if errors:
                raise ssdk.errors.SandboxTaskFailureError("\n".join(errors))

    def get_subtasks(self, task_type):
        tasks = ssdk.channel.channel.sandbox.list_tasks(parent_id=self.id)
        logging.debug("Task #{} children: {}".format(self.id, tasks))

        return [x for x in tasks if x.ctx.get(SUBTASK_TYPE) == task_type]

    def with_source_dir(self):
        return sdk_compat.get_source_dirs(self, sdk_compat.GET_ARCADIA_USE_ARC_VCS)[0]

    def get_main_subtasks(self):
        return self.get_subtasks('coverage_distbuild')

    def get_extra_subtasks(self):
        return self.get_subtasks('coverage_sandbox')

    def setup_finalization(self):
        self.stats = {}

        vault_owner = self.ctx.get(const.VAULT_OWNER, self.author)

        self.yt_token_path = "secret"
        fu.write_file(self.yt_token_path, self.get_vault_data(vault_owner, self.ctx.get('yt_token_vault_key')))

        if self.ctx.get('postgres_password_vault_key'):
            self.postgres_password = self.get_vault_data(vault_owner, self.ctx.get('postgres_password_vault_key'))
        else:
            self.postgres_password = None

        if self.ctx.get('postgres_prod_password_vault_key'):
            self.postgres_prod_password = self.get_vault_data(
                vault_owner, self.ctx.get('postgres_prod_password_vault_key')
            )
        else:
            self.postgres_prod_password = None

        self.revision = self.get_revision()
        self.snapshot = self.ctx[const.GRAPH_TIMESTAMP]

        self.coverage_wait_tool_warning_code = 10

        self.name_content_hash_file = self.get_log_filename('name_content_hash_file')
        self.build_name_content_hash_map()

    def setup_unified_agent_finalization(self):
        self.mega_uids_file = 'uids.list'
        for ident in self.ctx['uids_resources']:
            resource = ssdk.channel.channel.sandbox.get_resource(ident)
            if not resource.is_ready():
                logging.error('Resource {} is not ready, skip it'.format(ident))
                continue
            resouce_path = os.path.join(
                self.sync_resource(ident), os.path.basename(UPLOADED_UIDS_FILE_SALTED.format(''))
            )
            logging.debug('Id {}: resource path is {}'.format(ident, resouce_path))
            with open(self.mega_uids_file, 'a') as umf:
                with open(resouce_path) as rp:
                    umf.write(rp.read() + '\n')

    def run_wait_tool(self):
        name = os.path.basename(COVERAGE_WAIT_TOOL)
        res_id = self.get_tool_resource(COVERAGE_WAIT_TOOL)
        tool_dir = self.sync_resource(res_id)

        environment = os.environ.copy()
        prefix = 'datatable_' if self.ctx.get(const.COVERAGE_UNIFIED_AGENT_STRICT, False) else ''
        cmd = [
            os.path.join(tool_dir, name),
            '--yt-token-path',
            self.yt_token_path,
            '--yt-result',
            '{}/{}/{}/{}new_scheme_result'.format(YT_ROOT_PATH, self.revision, self.snapshot, prefix),
            '--uids-file',
            self.mega_uids_file,
            '--sid',
            self.ctx[const.COVERAGE_UNIFIED_AGENT_SID],
            '--start-table-name',
            self.ctx['start_table_name'],
            '--warning-percent',
            '98',
            '--warning-exit-code',
            str(self.coverage_wait_tool_warning_code),
            '--ttl',
            '30',
        ]
        start = time.time()
        p = ssdk.process.run_process(cmd, log_prefix=name, environment=environment, check=False)
        if p.returncode not in (0, self.coverage_wait_tool_warning_code) and self.ctx.get(
            const.COVERAGE_UNIFIED_AGENT_STRICT, False
        ):
            raise ssdk.errors.SandboxTaskFailureError(
                "Bad coverage wait tool return code %d. See logs for more information.", p.returncode
            )
        self.stats["{} walltime".format(name)] = time.time() - start
        return p.returncode

    def set_description(self):
        self.set_info(json.dumps(self.stats, sort_keys=True, indent=4))

    def build_name_content_hash_map(self):
        # Build name_content_hash_map to speed up coverage_spreader
        script_path = os.path.join(
            self.source_root, 'sandbox/projects/devtools/CoverageYaMakeSeminanti/build_name_content_hash_map.py'
        )
        cmd = [
            sys.executable,
            script_path,
            '--arcadia',
            self.source_root,
            '--output',
            self.name_content_hash_file,
            '--exts',
        ] + SRCS_EXTS
        logging.debug("Run cmd: %s", ' '.join(cmd))
        with open(self.name_content_hash_file + '.err', 'w') as afile:
            self.hash_map_proc = subprocess.Popen(cmd, stderr=afile)

    def get_tool_resource(self, artifact):
        tool_name = os.path.basename(artifact)
        task = self.get_subtasks(tool_name)[0]
        return ssdk.channel.channel.sandbox.list_resources(
            task_id=task.id,
            resource_type='ARCADIA_PROJECT',
            limit=1,
        )[0].id

    def merge_coverage(self, revision, snapshot):
        res_id = self.get_tool_resource(COVERAGE_MERGER)

        debug_args = []
        # debug_args = ['--dont-remove-data-tables']
        self.run_yt_tool(os.path.basename(COVERAGE_MERGER), res_id, revision, snapshot, cmd=debug_args)

    def update_last_link(self, revision, snapshot, source_root):
        cmd = [
            "python",
            os.path.join(source_root, "ya"),
            "tool",
            "yt",
            "--proxy=hahn",
            "link",
            "--force",
            "{}/{}/{}".format(YT_ROOT_PATH, revision, snapshot),
            "{}/last".format(YT_ROOT_PATH),
        ]

        env = os.environ.copy()
        env["YT_TOKEN_PATH"] = self.yt_token_path
        ssdk.process.run_process(cmd, log_prefix="ya_yt", environment=env)

    def spread_coverage(self, revision, snapshot, source_root):
        res_id = self.get_tool_resource(COVERAGE_SPREADER)

        env = {}
        dsts = []
        args = []
        if self.ctx.get('publish_coverage_arcanum') and self.postgres_password:
            env['YA_POSTGRES_PASSWORD'] = self.postgres_password
            dsts += ['arcanum']
        if self.ctx.get('publish_coverage_arcanum_prod') and self.postgres_prod_password:
            env['YA_POSTGRES_PROD_PASSWORD'] = self.postgres_prod_password
            dsts += ['arcanum-prod']
        if not dsts:
            logging.debug("Don't publish anything")
            return

        if not self.ctx.get('upload_dir_stat', False):
            args += ['--dont-update-dir-stat-table']

        cmd = (
            ['--source-root', source_root, '--name-content-hash-file', self.name_content_hash_file, '--destination']
            + dsts
            + args
        )

        self.run_yt_tool(os.path.basename(COVERAGE_SPREADER), res_id, revision, snapshot, cmd, env)
        return True

    def run_yt_tool(self, name, res_id, revision, snapshot, cmd=None, env=None):
        cmd = cmd or []
        logging.debug("Using %s from %s resource", name, res_id)
        tool_dir = self.sync_resource(res_id)

        environment = os.environ.copy()
        environment.update(env or {})

        base_cmd = [
            os.path.join(tool_dir, name),
            '--yt-token-path',
            self.yt_token_path,
            '--yt-root-path',
            '{}/{}'.format(YT_ROOT_PATH, revision),
            '--snapshot',
            snapshot,
            '--log-path',
            self.get_log_filename(name),
        ]
        start = time.time()
        ssdk.process.run_process(base_cmd + cmd, log_prefix=name, environment=environment)
        self.stats["{} walltime".format(name)] = time.time() - start

    def get_log_filename(self, prefix):
        logs_folder = ssdk.paths.get_logs_folder()
        filename = '{0}.log'.format(prefix)
        return ssdk.paths.get_unique_file_name(logs_folder, filename, limit=50)

    def wait_subtasks(self, tasks):
        statuses_to_wait = [
            Status.SUCCESS,
            Status.FAILURE,
            Status.DELETED,
            Status.EXCEPTION,
            Status.TIMEOUT,
            Status.STOPPED,
        ]
        self.wait_tasks(tasks, statuses_to_wait, wait_all=True)
        raise Exception('Unexpected behaviour')

    def obtain_extra_targets(self):
        logging.info("Obtaining extra tests targets with tag: %s", self.ctx['extra_tasks_test_tag'])

        if not self.ctx['extra_tasks_test_tag']:
            return []

        cmd = [
            sys.executable,
            os.path.join(self.source_root, 'ya'),
            '-v',
            'dump',
            'json-test-list',
            '-DAUTOCHECK=yes',
            '-DCLANG_COVERAGE=yes',
            '--skip-deps',
            '--target',
            os.path.join(self.source_root, self.get_target_dir()),
        ]
        logging.debug("Command: %s", cmd)

        filename = self.log_path("ya_dump") + '.graph'
        with open(filename, "w") as stdout, open(self.log_path("ya_dump") + ".err", "w", buffering=0) as stderr:
            proc = subprocess.Popen(cmd, stdout=stdout, stderr=stderr)
            proc.wait()

        if proc.returncode != 0:
            raise ssdk.errors.SandboxTaskFailureError(
                "Bad return code %d. See logs for more information.", proc.returncode
            )

        with open(filename) as afile:
            data = json.load(afile)

        targets = []
        for entry in data:
            dart_info = entry.get('dart_info', {})
            if self.ctx['extra_tasks_test_tag'] in dart_info.get('TAG', []):
                targets.append(TestInfo(dart_info))

        logging.info("Found targets for subtasks: %s", targets)
        return targets

    def setup_params(self):
        ts = int(time.time())
        self.ctx[const.GRAPH_TIMESTAMP] = str(ts)
        self.ctx['task_deadline_timestamp'] = str(ts + TASK_KILL_TIMEOUT)
        # We should use last rev from trunk if no revision specified, not head
        # because it might be revision from branch or something and TE won't work with it
        arc_url = self.ctx.get(const.ARCADIA_URL_KEY)
        parsed_arcadia_url = ssdk.svn.Arcadia.parse_url(arc_url)
        revision = parsed_arcadia_url.revision
        logging.debug("Parsed revision %s from arcadia_url: %s", revision, arc_url)

        if not revision:
            info = ssdk.svn.Arcadia.info(arc_url)
            logging.debug("arc_url svn info: %s", info)
            trunk_head = info["commit_revision"]
            self.ctx[const.ARCADIA_URL_KEY] = "{}@{}".format(arc_url, trunk_head)
        self.ctx['uids_resources'] = []
        self.ctx[const.COVERAGE_UNIFIED_AGENT_SID] = str(uuid.uuid4())
        logging.info(
            "Sid for coverage upload with unified agent scheme is: %s", self.ctx[const.COVERAGE_UNIFIED_AGENT_SID]
        )
        dt = datetime.datetime.utcnow() + datetime.timedelta(hours=3)  # logfeller uses MSK timezone for table names
        if dt.minute < 30:
            dt = dt.replace(minute=0)
        else:
            dt = dt.replace(minute=30)
        self.ctx['start_table_name'] = dt.strftime('%Y-%m-%dT%H:%M:00')
        # Set up ya token for distbuild pools
        if self.ctx.get(DistbuildPool.name):
            self.ctx['env_vars'] = "YA_TOKEN='$(vault:value:AUTOCHECK:devtools-coverage-ya-token)'"

    def create_large_coverage_subtasks(self, targets):
        build_execution_time = 2 * 3600  # 2h

        tasks = []
        for entry in targets:
            container = entry.get_container_id()
            # Don't use fuse for sandbox coverage tasks - there might no fuse libraries in the user's container
            fast_clang_coverage_merge = not bool(container)

            task = self.create_coverage_subtask(
                'Sandbox coverage task for {}'.format(entry.project_path),
                max_restarts=2,
                # task ctx
                build_system=const.SEMI_DISTBUILD_BUILD_SYSTEM,
                build_execution_time=build_execution_time,
                # 1h for sandbox machinery
                ya_timeout=build_execution_time + 3600,
                dns64=entry.use_dns64(),
                targets=entry.project_path,
                sandbox_container=container,
                privileged=entry.privileged,
                subtask_type='coverage_sandbox',
                test_size_filter='large',
                fast_clang_coverage_merge=fast_clang_coverage_merge,
                push_stats_to_solomon=self.ctx.get(PushStatsToSolomonParameter.name),
                solomon_common_labels=json.dumps({"contour": "sandbox", "partition": entry.project_path}),
                **{
                    const.COVERAGE_UNIFIED_AGENT: False,
                    const.COVERAGE_UNIFIED_AGENT_STRICT: False,
                }
            )
            tasks.append(task)

        return tasks

    def get_target_dir(self):
        return self.ctx.get('main_task_target', 'autocheck')

    def create_distbuild_coverage_tasks(self):
        target = self.get_target_dir()

        build_execution_time = 4 * 3600  # 4h
        partitions_count = 6

        return [
            self.create_coverage_subtask(
                'Distbuild coverage task for {} [{}/{}]'.format(target, partition, partitions_count),
                max_restarts=10,
                uids_filename_salt=partition,
                # task ctx
                build_system=const.DISTBUILD_FORCE_BUILD_SYSTEM,
                build_execution_time=build_execution_time,
                # 2h for sandbox machinery and downloading ya results
                ya_timeout=build_execution_time + (2 * 3600),
                targets=target,
                test_tag='-ya:force_sandbox',
                subtask_type='coverage_distbuild',
                # It's naive to believe that all targets in Arcadia are buildable
                check_ya_return_code=False,
                projects_partition_index=partition,
                projects_partitions_count=partitions_count,
                cache_test_results=True,
                push_stats_to_solomon=self.ctx.get(PushStatsToSolomonParameter.name),
                solomon_common_labels=json.dumps({"contour": "distbuild", "partition": str(partition)}),
            )
            for partition in range(partitions_count)
        ]

    def create_build_task(self, artifact):
        ctx = {
            'do_not_restart': False,
            'fail_on_any_error': False,
            'max_restarts': 2,  # SDK2 compatibility
            'kill_timeout': 1 * 60 * 60,
            # Task opts
            'aapi_fallback': True,
            'arts': artifact,
            'check_return_code': True,
            'clear_build': True,
            'definition_flags': '-DNO_STRIP=yes',
            'sandbox_tags': 'GENERIC&(LINUX_TRUSTY|LINUX_XENIAL)',
            'targets': os.path.dirname(artifact),
            'use_aapi_fuse': True,
            const.ARCADIA_URL_KEY: self.ctx.get(const.ARCADIA_URL_KEY),
            SUBTASK_TYPE: os.path.basename(artifact),
        }

        task = self.create_subtask(
            'YA_MAKE',
            'Build {}'.format(artifact),
            input_parameters=ctx,
            execution_space=20480,  # 20 GiB
            ram=10240,  # 10 GiB
        )

        logging.info('Created subtask %s with ctx: %s', task.id, json.dumps(ctx, indent=4, sort_keys=True))
        return task

    def create_coverage_subtask(self, descr=None, max_restarts=None, uids_filename_salt=None, **kwargs):
        ctx = copy.deepcopy(self.ctx)
        ctx.update(
            {
                # Support TemporaryError and task restart
                'do_not_restart': False,
                'fail_on_any_error': False,
                'max_restarts': max_restarts or 2,  # SDK2 compatibility
                'kill_timeout': TASK_KILL_TIMEOUT,
                'projects_partitions_count': kwargs.get('projects_partitions_count', 1),
                'projects_partition_index': kwargs.get('projects_partition_index', 0),
            }
        )
        ctx.update(kwargs)
        salted_uids_filename = UPLOADED_UIDS_FILE_SALTED.format(uids_filename_salt or "")

        if ctx.get(const.COVERAGE_UNIFIED_AGENT):
            if not os.path.exists(os.path.dirname(salted_uids_filename)):
                stp.mkdirp(os.path.dirname(salted_uids_filename))
            res = self.create_resource(
                (descr or "") + " uploaded uids file",
                os.path.dirname(salted_uids_filename),
                resource_types.OTHER_RESOURCE,
            )
            ctx.update(
                {
                    const.COVERAGE_UNIFIED_AGENT_UIDS_RESOURCE_ID: res.id,
                }
            )
            self.ctx['uids_resources'].append(res.id)
        ctx[const.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH] = salted_uids_filename

        task = self.create_subtask(
            'COVERAGE_YA_MAKE_TASK',
            '{}\n{}'.format(self.descr, descr or ''),
            max_restarts=max_restarts,
            input_parameters=ctx,
            execution_space=102400,  # 100 GiB
            ram=20480,  # 20 GiB
        )

        logging.info('Created subtask %s with ctx: %s', task.id, json.dumps(ctx, indent=4, sort_keys=True))
        return task

    def get_revision(self):
        parsed_arcadia_url = ssdk.svn.Arcadia.parse_url(self.ctx.get(const.ARCADIA_URL_KEY))
        return int(parsed_arcadia_url.revision)

    def push_stats_to_solomon(self):
        try:
            common_labels = {
                "contour": "meta",
            }
            token = self.get_vault_data('AUTOCHECK', 'devtools-ya-coverage-solomon-token')
            timestamp = int(self.ctx[const.GRAPH_TIMESTAMP])
            sensors = [
                {
                    'labels': {'sensor': k},
                    'ts': timestamp,
                    'value': v,
                }
                for k, v in self.stats.items()
            ]
            params = {
                'project': 'coverage',
                'cluster': 'arcadia',
                'service': 'nightly',
            }

            solomon.push_to_solomon_v2(token=token, params=params, common_labels=common_labels, sensors=sensors)
        except Exception as e:
            logging.exception('Failed to push stats to solomon: %s', e)

    def spawn_calc_coverage_stats(self):
        request = """
PRAGMA yt.InferSchema = '1';

SELECT test_type, count(*)
FROM hahn.`home/codecoverage/v1/{revision}/{snapshot}/data_table`
group by test_type order by test_type;
"""

        sqlfile = self.log_path('yql_stats') + '.sql'
        with open(sqlfile, 'w') as afile:
            afile.write(request.format(revision=self.revision, snapshot=self.ctx[const.GRAPH_TIMESTAMP]))

        outputfile = self.log_path('yql_stats') + '.json'

        cmd = [
            sys.executable,
            os.path.join(self.source_root, 'ya'),
            'yql',
            "--syntax-version",
            "1",
            "--input",
            sqlfile,
            "--output",
            outputfile,
            "--table-format",
            "json",
            "--no-column-types",
            "--no-column-names",
            "--fetch-full-results",
        ]
        logging.debug("Command: %s", cmd)

        env = os.environ.copy()
        env['YQL_TOKEN'] = self.get_vault_data('AUTOCHECK', 'devtools-ya-coverage-yql-token')

        with open(self.log_path("yql_stats") + ".out", "w") as stdout, open(
            self.log_path("yql_stats") + ".err", "w", buffering=0
        ) as stderr:
            proc = subprocess.Popen(cmd, env=env, stdout=stdout, stderr=stderr)

        ts = time.time()

        def wait():
            proc.wait()
            with open(outputfile) as afile:
                raw = afile.read()
            try:
                data = json.loads(raw.split('\n')[0])
            except Exception:
                raise Exception(raw)

            data = {k: int(v) for k, v in filter(None, data)}

            res = {"coverage raw {} entries".format(k): v for k, v in data.items()}
            res.update(
                {
                    'coverage entry types': len(data),
                    'calc stats wall time': int(time.time() - ts),
                }
            )
            return res

        return wait


__Task__ = CoverageYaMakeSeminanti
