from sandbox import sdk2
from sandbox.common.mds import MDS
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import environments


class CreateArcadiaHgBundle(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 100000

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.String("Hg repository URL", default="ssh://arcadia-hg.yandex-team.ru/arcadia.hg")
        ssh_user = sdk2.parameters.String("SSH user", required=True)
        ssh_key_vault_name = sdk2.parameters.String("SSH key vault name", required=True)
        ssh_key_vault_owner = sdk2.parameters.String("SSH key vault owner", required=True)

        with sdk2.parameters.Group("MDS parameters") as mds_block:
            mds_namespace = sdk2.parameters.String("MDS namespace", default="devtools")
            mds_token_vault_name = sdk2.parameters.String("MDS token vault name", default="MDS_DEVTOOLS_TOKEN")
            mds_token_vault_owner = sdk2.parameters.String("MDS token vault owner", default="YATOOL")
            mds_download_url_base = sdk2.parameters.String("MDS download URL base", default="https://storage.yandex-team.ru")
            mds_ttl = sdk2.parameters.Integer("MDS TTL", default=14)

        with sdk2.parameters.Output:
            mds_url = sdk2.parameters.String(
                "Hg bundle MDS url", default="", required=True
            )

    def clone_hg(self, dest):
        with sdk2.ssh.Key(self, self.Parameters.ssh_key_vault_owner, self.Parameters.ssh_key_vault_name):
            cmd = [
                "hg", "clone", "-U", "--uncompressed",
                "--ssh", "ssh -l {}".format(self.Parameters.ssh_user),
                self.Parameters.url, dest
            ]
            with sdk2.helpers.ProcessLog(self, logger="hg_clone") as pl:
                sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT).wait()

    def bundle_hg(self, repo, bundle):
        cmd = ["hg", "debugcreatestreamclonebundle", bundle]
        with sdk2.helpers.ProcessLog(self, logger="hg_bundle") as pl:
            sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT, cwd=repo).wait()

    def on_execute(self):
        environments.SandboxHgEnvironment(version="4.3.1y").prepare()

        repo = str(self.path("arcadia.hg"))
        self.clone_hg(repo)

        rev = sp.check_output(["hg", "heads", "--rev", "default", "--template", "{node}"], cwd=repo)

        bundle = str(self.path("stream.hg"))
        self.bundle_hg(repo, bundle)

        mds_token = sdk2.Vault.data(
            self.Parameters.mds_token_vault_owner,
            self.Parameters.mds_token_vault_name
        )

        mds_key = MDS().upload(
            path=bundle,
            mds_name="{}.hg".format(rev),
            tvm_service=None,
            token=mds_token,
            namespace=self.Parameters.mds_namespace,
            ttl=self.Parameters.mds_ttl
        )

        self.Parameters.mds_url = "{base}/get-{namespace}/{key}".format(
            base=self.Parameters.mds_download_url_base,
            namespace=self.Parameters.mds_namespace,
            key=mds_key
        )
