import logging
import os
import subprocess
from pipes import quote

from sandbox import sdk2 as sb

log = logging.getLogger(__name__)


def makedirs(dir):
    if not os.path.exists(dir):
        os.makedirs(dir)


def run(args):
    log.info('+ %s', ' '.join(map(quote, args)))
    subprocess.check_call(args)


def untar(archive, dir):
    makedirs(dir)
    run(['tar', '-C', dir, '-xf', archive])


def tar(archive, dir):
    run(['tar', '-C', dir, '-czf', archive, '.'])


def download(resource_id):
    return str(sb.ResourceData(resource_id).path)


def get_resource(resource_id):
    return list(sb.Resource.find(id=resource_id).limit(1))[0]


def attr_dict(o, attrs):
    return {k: getattr(o, k) for k in attrs if hasattr(o, k)}


def install_lld(root_dir, lld_path, no_gold=False):
    for dirpath, dirnames, filenames in os.walk(root_dir):
        for name in filenames:
            path = os.path.join(dirpath, name)
            if name == 'ld.gold':
                if no_gold:
                    os.unlink(path)
            elif name == 'ld':
                os.unlink(path)
                os.link(lld_path, path)


class REPLACE_LD_WITH_LLD(sb.Task):
    """
    Replace linker in resource with LLD. DEVTOOLS-3327
    """

    class Parameters(sb.Task.Parameters):
        compiler = sb.parameters.Resource('Compiler', required=True)
        lld = sb.parameters.Resource('LLD', required=True)
        no_gold = sb.parameters.Bool('Delete ld.gold')

    def on_execute(self):
        compiler_dir, lld_dir = map(os.path.abspath, ('compiler', 'lld'))
        untar(download(self.Parameters.compiler), compiler_dir)
        untar(download(self.Parameters.lld), lld_dir)
        install_lld(compiler_dir, lld_dir + '/lld', self.Parameters.no_gold)

        src = get_resource(self.Parameters.compiler.id)
        dst = sb.ResourceData(src.type(
            task=self,
            description=src.description + ' + LLD',
            path=src.path.name,
            arch=src.arch,
            **attr_dict(src, ('platform', 'version'))
        ))
        tar(str(dst.path), compiler_dir)
        dst.ready()
