# -*- coding: utf-8 -*-
import os
import shutil
import subprocess
import json

from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox.projects.common.build.parameters import ArcSecret
import sandbox.common.types.resource as ctr
from sandbox.projects.sandbox.resources import LXC_CONTAINER


PLUGIN_RELEASERS = [
    "perseus",
    "VSCODE-YANDEX",
]


class VSCODE_PLUGIN(sdk2.Resource):
    """ VS Code Arcadia Plugin resource """

    releasers = PLUGIN_RELEASERS
    releasable = True
    auto_backup = True
    any_arch = False

    version = sdk2.parameters.String("Plugin version from package.json", required=True)
    plugin_name = sdk2.parameters.String("Plugin name without publisher", required=True)


class BuildVSCodePlugin(sdk2.Task):
    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl()
        arc_secret = ArcSecret()
        version = sdk2.parameters.String('Version')

    def on_save(self):
        self.Requirements.container_resource = LXC_CONTAINER.find(
            state=ctr.State.READY,
            attrs={
                "image_type": "vscode_plugin_lxc",
            }
        ).first()

    def on_execute(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, use_arc_instead_of_aapi=True) as arcadia:
            plugin_path = os.path.join(arcadia, "devtools", "ide", "vscode-yandex-arc")
            with sdk2.helpers.ProcessLog(self) as l:
                subprocess.check_call(
                    ["npm", "install", "--save-dev", "--registry=http://npm.yandex-team.ru"],
                    cwd=plugin_path,
                    stdout=l.stdout,
                    stderr=l.stderr,
                )
                subprocess.check_call(
                    ["npm", "version", self.Parameters.version],
                    cwd=plugin_path,
                    stdout=l.stdout,
                    stderr=l.stderr
                )
                subprocess.check_call(
                    ["npm", "run", "build"],
                    cwd=plugin_path,
                    stdout=l.stdout,
                    stderr=l.stderr
                )
            with open(os.path.join(plugin_path, "package.json")) as f:
                package_json = json.load(f)
            version = package_json["version"]
            package_name = "vscode-yandex-arc-{}.vsix".format(version)
            shutil.copy(os.path.join(plugin_path, package_name), package_name)
        resource = VSCODE_PLUGIN(self, "VSCode Yandex", package_name)
        resource.version = version
        resource.plugin_name = "vscode-yandex-arc"
        resource.ttl = "inf"
        sdk2.ResourceData(resource).ready()


class TestVSCodePlugin(sdk2.Task):
    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl()
        arc_secret = ArcSecret()

    def on_save(self):
        self.Requirements.container_resource = LXC_CONTAINER.find(
            state=ctr.State.READY,
            attrs={
                "image_type": "vscode_plugin_lxc",
            }
        ).first()

    def on_execute(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, use_arc_instead_of_aapi=True) as arcadia:
            plugin_path = os.path.join(arcadia, "devtools", "ide", "vscode-yandex-arc")
            with sdk2.helpers.ProcessLog(self) as l:
                subprocess.check_call(
                    ["npm", "install", "--save-dev", "--registry=http://npm.yandex-team.ru"],
                    cwd=plugin_path,
                    stdout=l.stdout,
                    stderr=l.stderr,
                )
                subprocess.check_call(
                    ["xvfb-run", "-a", "npm", "test"],
                    cwd=plugin_path,
                    stdout=l.stdout,
                    stderr=l.stderr
                )
                # TODO test report
