# coding: utf-8

import copy
import json
import logging
import shutil

import sandbox.common as scommon
import sandbox.sandboxsdk as ssdk

from sandbox.common.types import client as ctc
from sandbox.common.types.task import Status, TaskStatus
from sandbox.projects.common import constants as const
from sandbox.projects.common.build import streaming_client
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.build.base_arcadia_project_task import BaseArcadiaProjectTask
from sandbox.projects.common.build.sdk import sdk_compat
from sandbox.sandboxsdk.channel import channel

TASK_STATE = 'task_state'
TASK_TYPE_NAME = 'task_type_name'
YT_CACHE_DIR = '//home/devtools/tmp/prettyboy'
YT_PROXY = 'hahn'
YT_STORE_CODEC = 'zstd08_1'
DEFAULT_HEATER_PLATFORM = 'linux_ubuntu_12.04_precise'

YA_TIMEOUT = int(1.5 * 3600)
KILL_TIMEOUT = YA_TIMEOUT + int(0.5 * 3600)


class TestingPlatformParameter(ssdk.parameters.SandboxStringParameter):
    name = 'testing_platform'
    description = ''
    default_value = ''


class HeaterPlatformParameter(ssdk.parameters.SandboxStringParameter):
    name = 'heater_platform'
    description = ''
    default_value = ''


class TestingPlatformTagsParameter(ssdk.parameters.SandboxStringParameter):
    name = 'testing_platform_tags'
    description = ''
    default_value = ''


class HeaterPlatformTagsParameter(ssdk.parameters.SandboxStringParameter):
    name = 'heater_platform_tags'
    description = ''
    default_value = ''


class YaMakeTestWithBuildHeater(BaseArcadiaProjectTask):

    type = 'YA_MAKE_TEST_WITH_BUILD_HEATER'
    execution_space = 1024
    client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT

    input_parameters = [
        TestingPlatformParameter,
        HeaterPlatformParameter,
        TestingPlatformTagsParameter,
        HeaterPlatformTagsParameter,
    ] + YaMakeTask.input_parameters

    def get_state(self):
        return self.ctx.get(TASK_STATE, 'init')

    def set_state(self, name):
        self.ctx[TASK_STATE] = name

    def on_execute(self):
        try:
            self.do_execute()
        except Exception as e:
            # (◞‸◟ㆀ)
            if not isinstance(e, scommon.errors.Wait):
                self.process_results(str(e))
            raise

    def do_execute(self):
        state = self.get_state()
        logging.debug("Current state: %s", state)
        if state == 'init':
            self.set_state('cross_compiled')
            self.setup_params()
            self.wait_subtasks([self.create_cross_compile_heater_task()])
        elif state == 'cross_compiled':
            task = self.get_cross_compile_heater_task()
            if task.status != TaskStatus.FINISHED:
                raise ssdk.errors.SandboxTaskFailureError("Cross compile heater subtask failed with status: {}".format(task.status))

            self.set_state('testing_done')
            self.wait_subtasks([self.create_testing_task()])
        elif state == 'testing_done':
            task = self.get_testing_task()
            if task.status != TaskStatus.FINISHED:
                raise ssdk.errors.SandboxTaskFailureError("Testing subtask failed with status: {}".format(task.status))

            self.set_state('all_done')
        else:
            raise ssdk.errors.SandboxTaskFailureError("Unknown state: {}".format(state))

    def on_finish(self):
        self.process_results()

    def process_results(self, error=None):
        try:
            if error:
                results = sdk_compat.build_error_result(self, self.id, error, "INTERNAL", "FAILED")
                results_path = "results.json"
                with open(results_path, "w") as afile:
                    json.dump({"results": results}, afile)
            else:
                resources = channel.sandbox.list_resources(
                    task_id=self.get_testing_task().id,
                    resource_type="TEST_ENVIRONMENT_JSON_V2",
                )
                results_path = self.sync_resource(resources[0])

            resource = self._create_resource(
                "results.json",
                "results.json",
                "TEST_ENVIRONMENT_JSON_V2",
            )
            shutil.copyfile(results_path, resource.abs_path())
            self.mark_resource_ready(resource.id)
        except Exception as e:
            logging.exception("Failed to setup resource: %s", e)

        # TODO remove code below when https://st.yandex-team.ru/DEVTOOLS-8610 is done

        streaming_link = self.ctx.get(const.STREAMING_REPORT_URL, None)
        streaming_check_id = self.ctx.get(const.STREAMING_REPORT_ID, None)

        if not streaming_link or not streaming_check_id:
            logging.debug("Stream won't be closed.")
            return

        client = streaming_client.StreamingClient(streaming_link, streaming_check_id, 0)

        if error:
            results = sdk_compat.build_error_result(self, self.id, error, "INTERNAL", "FAILED")
            logging.debug("Going to send error response: %s", json.dumps(results, indent=4, sort_keys=True))
            client.send_chunk(results)

        logging.debug("Going to close all streams")
        for size in ('small', 'medium', 'large'):
            client.close_stream_by_size('test', size)

        for tp in ('configure', 'build', 'style', 'test'):
            client.close_stream(tp)

        client.close()

    def setup_params(self):
        # Fixate revision for subtasks
        arc_url = self.ctx.get(const.ARCADIA_URL_KEY)
        parsed_arcadia_url = ssdk.svn.Arcadia.parse_url(arc_url)
        revision = parsed_arcadia_url.revision
        logging.debug("Parsed revision %s from arcadia_url: %s", revision, arc_url)

        if not revision:
            info = ssdk.svn.Arcadia.info(arc_url)
            logging.debug("arc_url svn info: %s", info)
            trunk_head = info["commit_revision"]
            self.ctx[const.ARCADIA_URL_KEY] = "{}@{}".format(arc_url, trunk_head)

    def get_subtasks(self, task_type):
        tasks = ssdk.channel.channel.sandbox.list_tasks(parent_id=self.id)
        logging.debug("Task #{} children: {}".format(self.id, tasks))

        tasks = [x for x in tasks if x.ctx.get(TASK_TYPE_NAME) == task_type]
        tasks = sorted(tasks, key=lambda x: x.id)
        return tasks

    def get_cross_compile_heater_task(self):
        return self.get_subtasks('cross_compile_heater')[-1]

    def get_testing_task(self):
        return self.get_subtasks('testing_task')[-1]

    def wait_subtasks(self, tasks):
        statuses_to_wait = [Status.SUCCESS, Status.FAILURE, Status.DELETED, Status.EXCEPTION, Status.TIMEOUT, Status.STOPPED]
        self.wait_tasks(tasks, statuses_to_wait, wait_all=True)
        raise Exception('Unexpected behaviour')

    def create_testing_task(self):
        sandbox_tags = 'OSX&GENERIC'
        if self.ctx['testing_platform_tags']:
            sandbox_tags = self.ctx['testing_platform_tags']
        return self.create_ya_make_subtask(
            self.ctx['testing_platform'],
            'Testing {}'.format(self.ctx['targets']),
            task_type_name='testing_task',
            max_restarts=None,
            # task ctx
            build_system=const.YA_MAKE_FORCE_BUILD_SYSTEM,
            keep_alive_all_streams=True,
            sandbox_tags=sandbox_tags,
            test=True,
            use_aapi_fuse=True,
            ya_yt_dir=YT_CACHE_DIR,
            ya_yt_proxy=YT_PROXY,
            ya_yt_put=False,
            ya_yt_store_codec=YT_STORE_CODEC,
            # yt_store_exclusive=True,
        )

    def create_cross_compile_heater_task(self):
        # TODO for linux
        # * build graph on the distbuild
        # * move task to multislot
        real_heater_platform = self.ctx['heater_platform'] or DEFAULT_HEATER_PLATFORM
        run_on_linux = real_heater_platform.startswith('linux')
        if run_on_linux:
            heater_build_system = const.DISTBUILD_FORCE_BUILD_SYSTEM
            retriable_displaced_build = True
        else:
            heater_build_system = const.YA_MAKE_FORCE_BUILD_SYSTEM
            retriable_displaced_build = False
        sandbox_tags = 'OSX&GENERIC' if not run_on_linux else self.ctx.get(const.SANDBOX_TAGS, None)  # should be better
        if self.ctx['heater_platform_tags']:
            sandbox_tags = self.ctx['heater_platform_tags']
        return self.create_ya_make_subtask(
            real_heater_platform,
            'Heater task for {}'.format(self.ctx['targets']),
            task_type_name='cross_compile_heater',
            max_restarts=100,
            # task ctx
            build_system=heater_build_system,
            keep_alive_all_streams=True,
            force_build_depends=True,
            test=False,
            sandbox_tags=sandbox_tags,
            # test filter should be dropped to avoid error when no tests run requested
            test_filters='',
            ya_yt_dir=YT_CACHE_DIR,
            ya_yt_max_cache_size='21990232555520',
            ya_yt_proxy=YT_PROXY,
            ya_yt_put=True,
            ya_yt_store=True,
            ya_yt_store_codec=YT_STORE_CODEC,
            use_aapi_fuse=True,
            retriable_displaced_build=retriable_displaced_build,
        )

    def create_ya_make_subtask(self, platform, descr, max_restarts, **kwargs):
        ctx = copy.deepcopy(self.ctx)
        task_type = 'YA_MAKE_MAPS_MOBILE_OSX' if (kwargs or {}).get('sandbox_tags', None) == 'CUSTOM_MAPS_MOBILE_DARWIN' else 'YA_MAKE'
        ctx.update({
            # Support TemporaryError and task restart
            'do_not_restart': False,
            'fail_on_any_error': False,
            # SDK2 compatibility
            'max_restarts': 2,
            'requested_arch': platform,
            'kill_timeout': KILL_TIMEOUT,
            'ya_timeout': YA_TIMEOUT,
        })
        ctx.update(kwargs)

        task = self.create_subtask(
            task_type,
            '{}\n{}'.format(self.descr, descr or ''),
            input_parameters=ctx,
            arch=platform,
            # execution_space=102400,  # 100 GiB
            # ram=20480,  # 20 GiB
            max_restarts=max_restarts,
        )

        logging.info('Created subtask %s with ctx: %s', task.id, json.dumps(ctx, indent=4, sort_keys=True))
        return task


__Task__ = YaMakeTestWithBuildHeater
