from sandbox import sdk2
from .resource_types import YMakeCacheDir, YMakeCacheTestResult
from .ymake_runner import ConfigYaMakeOptions
from .ymake_runner import DEFAULT_MAKE_ONLY_DIRS, DEFAULT_BUILD_TYPE, DEFAULT_BUILD_VARS, DEFAULT_HOST_PLATFORM_FLAGS
from .ymake_runner import DEFAULT_PARTITION_ARGUMENTS
import sandbox.sdk2.paths as sdk_paths
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import process

from sandbox.projects.common import binary_task
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common import task_env
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common as common

import json
import logging
import os
import tempfile


DESIGNATING_ATTRIBUTE_NAME = "ymake_heater_tasks"


def _get_svn_revision_and_arc_hash(arc_root):
    """
    Should be executed in Arc environment.
    """
    try:
        log_out = sp.check_output([Arc()._client_bin] + ['log', '-n', '1', '--json'], close_fds=True, cwd=arc_root)

        logging.info("arc log -n1 output: {}".format(log_out))

        j = json.loads(log_out)[0]
        svn_revision, arc_commit_id = (int(j['revision']), j['commit'])
        return svn_revision, arc_commit_id
    except Exception as exc:
        logging.error("Cannot read arc log -n1 --json: {}".format(exc))
        raise


def _get_ya_token(owner):
    secret_name = 'YA_TOKEN'
    diag = 'yt-store'
    try:
        logging.debug('Getting %s oauth token from vault: user=%s, secret_name=%s', diag, owner, secret_name)
        return sdk2.Vault.data(owner, secret_name)
    except common.errors.VaultError:
        logging.debug('Unable to fetch %s oauth token', diag)


def _get_custom_env(ya_token):
    custom_env = {}
    if ya_token:
        custom_env['YA_TOKEN'] = ya_token
        custom_env['YA_YT_TOKEN'] = ya_token
    custom_env['YA_CUSTOM_FETCHER'] = str(channel.channel.task.synchrophazotron)
    return custom_env


class YMakeCacheGroupParams(sdk2.Parameters):
    checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", default_value="arcadia:/arc/trunk/arcadia")
    heater_flags = sdk2.parameters.List('heater_flags', value_type=sdk2.parameters.String,
                                        default=['-xx', '-xF'])
    ymake_cache_kind = sdk2.parameters.String('ymake_cache_kind (should be consistent with heater_flags)', default="parser")
    ymake_cache_ttl = sdk2.parameters.Integer("ymake_cache_ttl", default=2)

    with sdk2.parameters.Group("testenv/jobs/autocheck/BaseBuildTrunkMeta.py") as testenv_block:

        autocheck_make_only_dirs = sdk2.parameters.List("autocheck_make_only_dirs (change to debug)", value_type=sdk2.parameters.String,
                                                        default=DEFAULT_MAKE_ONLY_DIRS)
        autocheck_build_type = sdk2.parameters.String("autocheck_build_type", default=DEFAULT_BUILD_TYPE)
        autocheck_build_vars = sdk2.parameters.List("autocheck_build_vars", value_type=sdk2.parameters.String,
                                                    default=DEFAULT_BUILD_VARS)
        host_platform_flags = sdk2.parameters.List("host_platform_flags", value_type=sdk2.parameters.String,
                                                   default=DEFAULT_HOST_PLATFORM_FLAGS)

    with sdk2.parameters.Group('development') as development:
        build_ya_bin = sdk2.parameters.Bool('build_ya_bin', default=False)


class YMakeCacheParams(YMakeCacheGroupParams):
    with sdk2.parameters.Group("Set by parent task") as parent_block:
        autocheck_config_path = sdk2.parameters.String("autocheck_config_path", default="")
        projects_partition_index = sdk2.parameters.Integer("projects_partition_index", default=0)
        projects_partitions_count = sdk2.parameters.Integer("projects_partitions_count", default=1)


class YmakeCacheHeaterChild(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """ Collect ymake caches for single 'ya make' run with selected autocheck-related arguments """

    class Requirements(task_env.BuildLinuxRequirements):
        ram = 16*1024
        disk_space = 32000

    class Parameters(YMakeCacheParams):
        with sdk2.parameters.Group('Internal parameters') as internal_parameters:
            bin_params = binary_task.LastBinaryReleaseParameters()

    class Context(sdk2.Task.Context):
        arc_root = None

    @property
    def binary_executor_query(self):
        return {
            'attrs': {DESIGNATING_ATTRIBUTE_NAME: True, 'released': self.Parameters.binary_executor_release_type},
            'owner': 'YATOOL',
            'state': [ctr.State.READY],
        }

    def on_execute(self):
        self._ya_token = _get_ya_token(self.owner)

        super(YmakeCacheHeaterChild, self).on_execute()
        try:
            parsed_url = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
            if not parsed_url.trunk:
                raise common.errors.TaskFailure("Works with trunk only at this moment")
        except Exception as exc:
            logging.error("Exception caught: %s", exc)
            raise common.errors.TaskFailure(exc)

        svn_ref = 'r' + str(parsed_url.revision) if parsed_url.revision else None
        try:
            with Arc().mount_path(None, svn_ref, fetch_all=False) as arc_root:
                self.Context.arc_root = arc_root

                svn_revision, arc_commit_id = _get_svn_revision_and_arc_hash(arc_root)

                self._run_ya_make(arc_commit_id, svn_revision)
        except Exception as exc:
            logging.error("Exception caught: %s", exc)
            raise common.errors.TaskFailure(exc)

    def _run_ya_make(self, arc_commit_id, svn_revision):
        resource = YMakeCacheDir(
            task=self,
            description="Cache {} {} {} {} {}".format(self.Parameters.autocheck_config_path,
                                                      self.Parameters.projects_partition_index,
                                                      self.Parameters.projects_partitions_count,
                                                      self.Parameters.ymake_cache_kind,
                                                      self.Parameters.autocheck_make_only_dirs),
            path="caches.tar.zst",
            arc_commit_id=arc_commit_id,
            svn_revision=int(svn_revision),
            autocheck_config_path=self.Parameters.autocheck_config_path,
            projects_partition_index=self.Parameters.projects_partition_index,
            projects_partitions_count=self.Parameters.projects_partitions_count,
            ymake_cache_kind=self.Parameters.ymake_cache_kind,
            autocheck_make_only_dirs=self.Parameters.autocheck_make_only_dirs,
            ttl=self.Parameters.ymake_cache_ttl,
            parent_task_executor_released=self.Parameters.binary_executor_release_type,
            backup_task=True,
        )

        data = sdk2.ResourceData(resource)

        source_root = self.Context.arc_root
        build_root = tempfile.mkdtemp()
        output_root = tempfile.mkdtemp()
        cache_archive = str(data.path)
        try:
            if self.Parameters.build_ya_bin:
                self._build_ya_bin(source_root)

            with sdk2.helpers.ProcessLog(self, logger='ymake-log') as pl:
                cmd = ConfigYaMakeOptions(source_root, output_root, build_root, cache_archive, self.Parameters).gen_cmd(use_ya_bin=self.Parameters.build_ya_bin)
                if self._ya_token:
                    cmd += ['--yt-store']
                with process.CustomOsEnviron(_get_custom_env(self._ya_token)):
                    proc = sp.Popen(
                        cmd,
                        close_fds=True, cwd=source_root,
                        stdout=pl.stdout, stderr=pl.stderr
                    )
                    proc.wait()
                    if proc.returncode:
                        logging.error("Heater failed: {}".format(cmd))
                        raise common.errors.TaskFailure("Heater failed: {}".format(cmd))

            # Mark resource as ready:
            data.ready()
        finally:
            sdk_paths.remove_path(build_root)
            sdk_paths.remove_path(output_root)

    def _build_ya_bin(self, source_root):
        with sdk2.helpers.ProcessLog(self, logger='ya-bin-build') as pl:
            cmd = [os.path.join(source_root, 'ya'), "-v", "--no-report", "make", "-r", os.path.join("devtools", "ya", "bin")]
            if self._ya_token:
                cmd += ['--yt-store']
            with process.CustomOsEnviron(_get_custom_env(self._ya_token)):
                proc = sp.Popen(
                    cmd,
                    close_fds=True, cwd=source_root,
                    stdout=pl.stdout, stderr=pl.stderr
                )
                proc.wait()
                if proc.returncode:
                    logging.error("Heater failed, ya-bin not built: {}".format(cmd))
                    raise common.errors.TaskFailure("Heater failed, ya-bin not built: {}".format(cmd))


class YmakeCacheHeater(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """ A task which creates ymake caches"""

    class Requirements(task_env.TinyRequirements):
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX
        disk_space = 2048

    class Parameters(YMakeCacheGroupParams):
        configs = sdk2.parameters.JSON("configs", default=DEFAULT_PARTITION_ARGUMENTS)
        with sdk2.parameters.Group('Internal parameters') as internal_parameters:
            bin_params = binary_task.LastBinaryReleaseParameters()

    class Context(sdk2.Task.Context):
        tasks = []

    @property
    def binary_executor_query(self):
        return {
            'attrs': {DESIGNATING_ATTRIBUTE_NAME: True, 'released': self.Parameters.binary_executor_release_type},
            # 'owner': 'AUTOCHECK_ACCEPTANCE',
            'state': [ctr.State.READY],
        }

    def _create_child_task(self, path, index, count):
        task = YmakeCacheHeaterChild(
            self,
            description="Generate ymake caches for {}, config={}, {}/{}".format(str(self.Parameters.checkout_arcadia_from_url), path, index, count)
        )

        task.Parameters.checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        task.Parameters.autocheck_config_path = path
        task.Parameters.projects_partition_index = index
        task.Parameters.projects_partitions_count = count

        task.Parameters.autocheck_build_type = self.Parameters.autocheck_build_type
        task.Parameters.ymake_cache_ttl = self.Parameters.ymake_cache_ttl
        task.Parameters.host_platform_flags = self.Parameters.host_platform_flags
        task.Parameters.autocheck_build_vars = self.Parameters.autocheck_build_vars
        task.Parameters.autocheck_make_only_dirs = self.Parameters.autocheck_make_only_dirs
        task.Parameters.heater_flags = self.Parameters.heater_flags
        task.Parameters.ymake_cache_kind = self.Parameters.ymake_cache_kind
        task.Parameters.build_ya_bin = self.Parameters.build_ya_bin

        task.Parameters.binary_executor_release_type = self.Parameters.binary_executor_release_type

        task.save()
        return task

    def _create_all_children(self):
        tasks = []
        for conf, cnt in self.Parameters.configs:
            for ind in range(cnt):
                task = self._create_child_task(conf, ind, cnt)
                task.enqueue()
                self.Context.tasks.append(task.id)
                tasks.append(task)
        raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        super(YmakeCacheHeater, self).on_execute()
        with self.memoize_stage.run_tests:
            if self.Context.tasks == []:
                self._create_all_children()

        with self.memoize_stage.check_results:
            if not all(task.status == ctt.Status.SUCCESS for task in self.find(parent_id=self.id)):
                raise common.errors.TaskFailure('Some of child tasks failed')

        logging.info('Subtasks have finished')


class YmakeCacheTestChild(sdk2.Task):
    """ Runs some test with tag ya:manual """

    class Requirements(task_env.BuildLinuxRequirements):
        ram = 16*1024
        disk_space = 32000

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", default_value="arcadia:/arc/trunk/arcadia")
        apply_patch = sdk2.parameters.String("apply_patch", default="")

        test_directory = sdk2.parameters.String("test_directory", default="")
        test_pattern = sdk2.parameters.String("test_pattern", default="")
        test_params = sdk2.parameters.List("test_params", value_type=sdk2.parameters.String, default=[])

        cleanup_result = sdk2.parameters.Bool('cleanup_results', default=True)

    class Context(sdk2.Task.Context):
        arc_root = None
        svn_revision = None
        arc_commit_id = None

    def on_execute(self):
        self._ya_token = _get_ya_token(self.owner)
        try:
            parsed_url = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
            if not parsed_url.trunk:
                raise common.errors.TaskFailure("Works with trunk only at this moment")
        except Exception as exc:
            logging.error("Exception caught: %s", exc)
            raise common.errors.TaskFailure(exc)

        svn_ref = 'r' + str(parsed_url.revision) if parsed_url.revision else None
        try:
            with Arc().mount_path(None, svn_ref, fetch_all=False) as arc_root:
                self.Context.arc_root = arc_root
                self.Context.svn_revision, self.Context.arc_commit_id = _get_svn_revision_and_arc_hash(arc_root)
                self._apply_patch(arc_root)
                self._run_ya_make()
        except Exception as exc:
            logging.error("Exception caught: %s", exc)
            raise common.errors.TaskFailure(exc)

    def _apply_patch(self, arc_root):
        if not self.Parameters.apply_patch:
            return

        logging.info('Applying patch {}'.format(self.Parameters.apply_patch))

        patch_store = tempfile.mkdtemp()
        patch_path, is_zipatch = Arcadia.fetch_patch(self.Parameters.apply_patch, patch_store)
        with open(patch_path, 'r') as f:
            logging.info('Patch {}'.format(f.read(50000)))

        Arcadia.apply_patch_file(arc_root, patch_path, is_zipatch)

        logging.info('Applied patch')

    def _run_ya_make(self):
        source_root = self.Context.arc_root
        build_root = tempfile.mkdtemp()
        test_output_res = YMakeCacheTestResult(
            task=self,
            description="Test output {} {} {}".format(self.Parameters.checkout_arcadia_from_url,
                                                      self.Parameters.test_directory,
                                                      self.Parameters.test_pattern),
            path="test_output",
            test_directory=self.Parameters.test_directory,
            test_pattern=self.Parameters.test_pattern)

        data = sdk2.ResourceData(test_output_res)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        try:
            for mode in (True, False):
                with sdk2.helpers.ProcessLog(self, logger='ymake-log-{}'.format(mode)) as pl:
                    cmd = self._ymake_cmd(source_root, build_root, str(data.path), build_only=mode)
                    if self._ya_token:
                        cmd += ['--yt-store']
                    with process.CustomOsEnviron(_get_custom_env(self._ya_token)):
                        proc = sp.Popen(
                            cmd,
                            close_fds=True, cwd=source_root,
                            stdout=pl.stdout, stderr=pl.stderr
                        )
                        proc.wait()
                        if self.Parameters.cleanup_result:
                            self._cleanup_results(str(data.path), proc.returncode)

                        # build_only only is True then ignore exception
                        if proc.returncode and not mode:
                            logging.error("Test failed: {}".format(cmd))
                            raise common.errors.TaskFailure("Test failed: {}".format(cmd))

        except Exception as e:
            logging.error("Caught exception during test execution: {}".format(e))
            raise
        finally:
            data.ready()
            sdk_paths.remove_path(build_root)

    def _ymake_cmd(self, arc_root, build_root, output_root, build_only):
        return [os.path.join(arc_root, 'ya'), "-v", "--no-report", "make", "-tA", "-r", "--threads", "16"] + self.Parameters.test_params + [
            "-B", build_root, "--no-src-links", "-o", output_root, '--cache-tests',
            self.Parameters.test_directory, "-F", self.Parameters.test_pattern] + \
            (['--dist', '--download-artifacts',  '-L'] if build_only else [])

    def _cleanup_results(sefl, output_root, returncode):
        def filter(f):
            if os.path.getsize(ff) < 10 * 1024 * 1024:
                return False
            if f == "stderr":
                return False
            if returncode and f.startswith('ya-bin.err'):
                return False
            if returncode and f.startswith('generating.log'):
                return False

            return True

        for path, dirs, files in os.walk(output_root):
            for f in files:
                ff = os.path.join(path, f)
                if filter(f):
                    os.remove(ff)


class YmakeCacheTest(sdk2.Task):
    """ ya make -tA devtools/ya/build/tests/build_graph_cache_autocheck --test-tag ya:manual \
        -F test_sandbox_resource.py::TestAutoCheck::test_build_graph_cache[config*]
    """

    class Requirements(task_env.TinyRequirements):
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX
        disk_space = 2048

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", default_value="arcadia:/arc/trunk/arcadia")
        apply_patch = sdk2.parameters.String("apply_patch", default="")

        configs = sdk2.parameters.JSON("configs", default=DEFAULT_PARTITION_ARGUMENTS)

        test_directory = sdk2.parameters.String("test_directory", default="devtools/ya/build/tests/build_graph_cache_autocheck")
        test_pattern = sdk2.parameters.String("test_pattern", default="test_sandbox_resource.py::TestAutoCheck::test_build_graph_cache[config{}]")
        test_params = sdk2.parameters.List("test_params", value_type=sdk2.parameters.String, default=[])

        cleanup_result = sdk2.parameters.Bool('cleanup_results', default=True)

    class Context(sdk2.Task.Context):
        tasks = []

    def _create_child_task(self, config_index, path, index, count):
        task = YmakeCacheTestChild(
            self,
            description="Check ymake caches in test #{}, config={}, {}/{}".format(
                config_index, path, index, count)
        )
        task.Parameters.checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        task.Parameters.apply_patch = self.Parameters.apply_patch

        task.Parameters.test_directory = self.Parameters.test_directory
        task.Parameters.test_pattern = self.Parameters.test_pattern.format(config_index)
        task.Parameters.test_params = self.Parameters.test_params
        task.Parameters.cleanup_result = self.Parameters.cleanup_result
        task.save()
        return task

    def _create_all_children(self):
        tasks = []
        config_index = 0
        for conf, cnt in self.Parameters.configs:
            for ind in range(cnt):
                task = self._create_child_task(config_index, conf, ind, cnt)
                task.enqueue()
                self.Context.tasks.append(task.id)
                tasks.append(task)

                config_index += 1
        raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.run_tests:
            if self.Context.tasks == []:
                self._create_all_children()

        with self.memoize_stage.check_results:
            if not all(task.status == ctt.Status.SUCCESS for task in self.find(parent_id=self.id)):
                raise common.errors.TaskFailure('Some of child tasks failed')

        logging.info('Subtasks have finished')
