import hashlib
import itertools
import json
import six
import logging

import sandbox.projects.common.constants as consts


DEFAULT_SUBTASKS_COUNT = 1
PERFTEST_SUBTASKS_COUNT = 3

logger = logging.getLogger(__name__)


def get_suite_info(task):
    return json.loads(task.Parameters.expected_test_info)


def get_suite_toolchain(task):
    return get_suite_info(task).get("toolchain", "TOOLCHAIN_IS_NOT_SPECIFIED")


def get_suite_requirements(task, default={}):
    return get_suite_info(task).get("requirements", default)


def get_suite_tags(task, default=[]):
    return get_suite_info(task).get("tags", default)


def get_subtasks_count(task):
    import test.const as tconst

    if tconst.YaTestTags.PerfTest in get_suite_tags(task):
        return PERFTEST_SUBTASKS_COUNT
    return DEFAULT_SUBTASKS_COUNT


def get_suite_type(task):
    return get_suite_info(task).get("suite_name")


def get_suite_hid(task):
    return get_suite_info(task).get("suite_hid")


def get_suite_id(task):
    return get_suite_info(task).get("suite_id")


def normalize_size_requirement(val):
    # Returns size in bytes. Normalize requirements which are specified in GB
    if val < 1000:
        return val * 1024**3
    return val


def match_resource(resources, **kwargs):
    for res in resources:
        if all(res.get(k) == v for k, v in kwargs.items()):
            return res


def get_task_resource(task, **kwargs):
    resources = task.resources
    resources_info = resources.read()['items']
    return match_resource(resources_info, **kwargs)


def html_to_markup(text):
    if not text:
        return ''

    class Parser(six.moves.html_parser.HTMLParser):
        def __init__(self):
            six.moves.html_parser.HTMLParser.__init__(self)
            self.text = six.StringIO()
            self.a = False

        def handle_starttag(self, tag, attrs):
            if tag == 'a':
                self.text.write("[[path]]{}".format(dict(attrs).get('href')))
                self.a = True

        def handle_endtag(self, tag):
            if tag == 'a':
                self.a = False
            self.text.write("[[rst]]")

        def handle_data(self, data):
            if not self.a:
                self.text.write(data)

    parser = Parser()
    parser.feed(text)
    return parser.text.getvalue()


def get_hid(data):
    import cityhash

    return cityhash.hash64(''.join(data))


def get_id(data, prefix=None):
    md5 = hashlib.md5()
    for x in data:
        md5.update(x)
    res = md5.hexdigest()

    if prefix:
        return '{}-{}'.format(prefix, res)
    return res


def split_tags(tags):
    def real(e):
        return list(filter(None, e))

    res = []
    for p1 in real(tags.split("|")):
        p2 = list(x.strip() for x in real(p1.split("&")))
        res.append(p2)
    return res


def join_tags(groups):
    return join_tags_union(join_tags_inter(g) for g in groups)


def join_tags_inter(tags):
    return "&".join(filter(None, tags))


def join_tags_union(tags):
    return "|".join(filter(None, tags))


def parse_tags(tags):
    tag_params = {}
    sandbox_tags = []

    for x in tags:
        if not x.startswith("sb:"):
            continue

        tag_line = x[3:]
        for group in split_tags(tag_line):
            real_tags = []
            for tag in group:
                if '=' in tag:
                    k, v = tag.split('=', 1)
                    tag_params[k] = v
                else:
                    real_tags.append(tag)

            sandbox_tags.append(real_tags)

    return sandbox_tags, tag_params


def union(data, fun):
    return sorted(set(itertools.chain.from_iterable(map(fun, data))))


def get_task_url(task_id):
    return "https://sandbox.yandex-team.ru/task/{}".format(task_id)


def get_resource_url(res_id):
    return "https://proxy.sandbox.yandex-team.ru/{}".format(res_id)


def get_suites(data):
    for x in data:
        if x.get('suite'):
            yield x


def read_json_safe(path, default=None):
    try:
        with open(path) as afile:
            return json.load(afile)
    except Exception as e:
        logger.error("Failed to load %s: %s", path, e)
        return default


def switch_arcadia_url_to_arc(context):
    if context[consts.ARCADIA_URL_KEY].startswith('arcadia:/arc/trunk/arcadia@'):
        context[consts.ARCADIA_URL_KEY] = context[consts.ARCADIA_URL_KEY].replace(
            'arcadia:/arc/trunk/arcadia@', 'arcadia-arc:/#r'
        )
