import collections
import copy
import logging
import six


logger = logging.getLogger(__name__)


def get_test_links_snippet(test, task_link):
    lines = []
    default_links = {"task": [task_link]}
    for name, val in (test.get("links", default_links) or default_links).items():
        lines.append("{} ([[path]]{}[[rst]])".format(name, val[0]))
    return "\n".join(lines)


def median(values):
    values = sorted(values)
    half, odd = divmod(len(values), 2)
    if odd:
        return values[half]
    return (values[half - 1] + values[half]) / 2.0


def merge_tests(tests, result_paths):
    if len(tests) == len(result_paths) == 1:
        return tests[0]

    merged = copy.deepcopy(tests[0])
    test_by_path_mapping = {t["sandbox_task"]: t for t in tests}

    merged["links"] = {}
    for i, links in enumerate([t.get("links", {}) for t in tests]):
        for name, val in links.items():
            merged["links"]["{}_{}".format(name, i + 1)] = val

    aggregated_metrics = {}
    for test_metrics in [t.get("metrics", {}) for t in tests]:
        for metric_name, metric_value in test_metrics.items():
            if metric_name not in aggregated_metrics:
                aggregated_metrics[metric_name] = []
            aggregated_metrics[metric_name].append(metric_value)

    merged["metrics"] = {}
    for metric_name, metric_values in aggregated_metrics.items():
        parsed = six.moves.urllib.parse.urlparse(metric_name)
        metric_name = parsed.path
        metric_meta = six.moves.urllib.parse.parse_qs(parsed.query)
        aggregating_funcs = {
            "avg": lambda l: sum(l) / float(len(l)),
            "min": min,
            "max": max,
            "sum": sum,
            "median": median,
        }
        aggregating_func = aggregating_funcs.get(metric_meta.get("aggr", ["avg"])[0])
        merged["metrics"][metric_name] = aggregating_func(metric_values)

    statuses = collections.defaultdict(int)
    ok_status = "{}_{}".format("OK", None)
    for t in tests:
        statuses["{}_{}".format(t["status"], t.get("error_type"))] += 1

    if len(statuses.keys()) > 1 and statuses[ok_status] < 1 or len(test_by_path_mapping) < 1:
        merged["status"] = "FAILED"
        merged["error_type"] = "REGULAR"
    elif len(statuses.keys()) > 1:
        # otherwise tests[0] will bring the proper status to the merged test
        merged["status"] = "OK"
        if "error_type" in merged:
            del merged["error_type"]

    def get_color(val):
        return {
            "OK": "good",
            "FAILED": "bad",
        }.get(val, "alt4")

    lines = []
    for result_path in result_paths:
        if result_path in test_by_path_mapping:
            test = test_by_path_mapping[result_path]
            status = test["status"]
            msg = "[[{}]]{}[[rst]]:\n{}".format(get_color(status), status, get_test_links_snippet(test, result_path))
        else:
            msg = "[[bad]]NO_DATA[[rst]]: [[path]]{}[[rst]]".format(result_path)
        lines.append(msg)

    merged["rich-snippet"] = "\n".join(lines)
    return merged


def merge(tests_results):
    """
    Merges test results
    :param tests_results: test results map {<result task link>: {tests from the task}
    :param quorum_count count of results enough to form test result
    :return: merged tests
    """
    aggregated = {}
    for sandbox_task, tests in tests_results.items():
        for record in tests:
            if record["type"] == "test":
                record["sandbox_task"] = sandbox_task
                test_id = record["id"]
                if test_id not in aggregated:
                    aggregated[test_id] = []
                aggregated[test_id].append(record)

    return [merge_tests(tests, tests_results.keys()) for tests in aggregated.values()]
