import json
import logging
import os

from . import helpers
from . import merger
from . import metrics
from . import native

import sandbox.common.types.task as ctt
import sandbox.projects.common.build.sdk.sdk_compat as sdk_compat
import sandbox.projects.resource_types as resource_types
import sandbox.sdk2 as sdk2


logger = logging.getLogger(__name__)


def process_start(parent):
    report_start(parent)


def process_results(parent, error=None):
    if error:
        snippet = "Exception during subtasks spawning:\n{}".format(error)
        results = sdk_compat.build_error_result(parent, parent.id, snippet, 'REGULAR', 'FAILED')
    else:
        results = obtain_results(parent)

    res_data = {
        "results": results,
        "static_values": {},
    }

    create_report_resource(parent, res_data)
    report_results(parent, res_data)


def obtain_results(parent):
    results_map = {}
    task_map = {}

    for tid in parent.Context.child_task_ids:
        task = parent.server.task[tid]
        res = get_results(parent, task, tid)

        task_map[tid] = task
        results_map[tid] = res

    # Original results from child task contain only build and configure results - generate a suite for native build
    if parent.Parameters.native_target:
        assert len(results_map) == 1, "multiple subtasks are not supported"
        tid, res = results_map.items()[0]
        results_map[tid] = native.convert_native_results(res, parent.Parameters.native_specification)

    for tid, task in task_map.items():
        results_map[tid] = saturate_results(task, tid, results_map[tid])

    if len(results_map) == 1:
        return results_map.values()[0]
    return merger.merge(results_map)


def report_results(parent, results, stream_partition=0):
    if parent.Parameters.streaming_link and parent.Parameters.streaming_check_id:
        sdk_compat.report_streaming_results(
            parent.Parameters.streaming_link,
            parent.Parameters.streaming_check_id,
            results,
            stream_partition,
        )

    def impl(storage_client):
        storage_client.send_message(parent.Parameters.ci_task_id, stream_partition, results, True)

    call_storage(parent, impl)


def report_start(parent, stream_partition=0):
    def impl(storage_client):
        storage_client.send_start(parent.Parameters.ci_task_id, stream_partition)

    call_storage(parent, impl)


def call_storage(parent, method):
    # CI-2688 - ALWAYS THROWS ERRORS AFTER MIGRATING TO CI
    throw_errors = not (parent.Parameters.streaming_link and parent.Parameters.streaming_check_id)
    if parent.Parameters.report_to_ci:
        try:
            from testenv.autocheck.ci_proxy_client import ci_proxy_client

            storage_client = ci_proxy_client.CIProxyClient(
                parent.Parameters.ci_endpoints.split(','),
                parent.Parameters.ci_check_id,
                parent.Parameters.ci_iteration_number,
                'HEAVY',
                parent.Parameters.ci_type,
                parent.id,  # Sandbox task_id
            )
            method(storage_client)
        except Exception as e:
            logging.exception('Exception occurred while calling CI Client')
            if throw_errors:
                raise Exception('Exception occurred while calling CI Client', e)


def create_report_resource(parent, results):
    res = sdk2.Resource[resource_types.TEST_ENVIRONMENT_JSON_V2](
        parent,
        "TestEnv Json",
        "results.json",
        attrs={'ttl': 30},
    )

    resdata = sdk2.ResourceData(res)
    data = json.dumps(
        results,
        indent=4,
        sort_keys=True,
    )
    resdata.path.write_bytes(data)
    resdata.ready()

    return str(resdata.path)


def saturate_results(task, tid, results):
    extra = {
        "links": {
            "sandbox_task": [
                helpers.get_task_url(tid),
            ]
        },
        "metrics": {},
    }

    res = helpers.get_task_resource(task, type='BUILD_OUTPUT_HTML', state='READY')
    if res:
        extra["links"]["ya_make_html"] = [helpers.get_resource_url(res['id'])]

    res = helpers.get_task_resource(task, type='BUILD_STATISTICS', state='READY')
    if res:
        path = sync_resource(res['id'])

        cache_hit = helpers.read_json_safe(os.path.join(path, "cache-hit.json"))
        if cache_hit:
            extra["metrics"]["sandbox.task.ya.cache_hit_percent"] = float(cache_hit.get('cache_hit', 0))

        critical_path = helpers.read_json_safe(os.path.join(path, "critical-path.json"))
        if critical_path:
            total = sum(n.get('elapsed', 0) for n in critical_path)
            extra["metrics"]["sandbox.task.ya.critical_path_seconds"] = total / 1000

    ya_build_threads = metrics.get_ya_build_threads(task)
    if ya_build_threads:
        extra["metrics"]["sandbox.task.ya.build_threads"] = ya_build_threads
    ya_make_task_duration = metrics.get_ya_make_task_duration(task)
    if ya_make_task_duration:
        extra["metrics"]["sandbox.task.ya_make_task_seconds"] = ya_make_task_duration
    build_time, test_time = metrics.get_build_and_test_times(task)
    if build_time:
        extra["metrics"]["sandbox.task.build.ya.walltime_seconds"] = build_time
    if test_time:
        extra["metrics"]["sandbox.task.test.ya.walltime_seconds"] = test_time

    logger.debug("Extra suite info: %s", json.dumps(extra, indent=4, sort_keys=True))

    for suite in helpers.get_suites(results):
        for k, v in extra.items():
            if k in suite:
                suite[k].update(v)
            else:
                suite[k] = v

    return results


def sync_resource(rid):
    return str(sdk2.ResourceData(sdk2.Resource[rid]).path)


def get_results(parent, task, task_id):
    task_info = task.read()

    if task_info.get('status') == ctt.Status.SUCCESS:
        res = helpers.get_task_resource(task, type='TEST_ENVIRONMENT_JSON_V2', state='READY')
        if res:
            filename = sync_resource(res['id'])
            try:
                with open(filename) as afile:
                    results = json.load(afile)['results']
            except Exception as e:
                logger.exception("Failed to load results: %s", e)
                snippet = "[[bad]]Failed to load {} (results.json): {}[[rst]]".format(res['http']['proxy'], e)
                return sdk_compat.build_error_result(parent, task_id, snippet, 'REGULAR', 'FAILED')

            if results:
                return results

            snippet = "[[bad]]Failed to load {} (results.json): it doesn't contain any data[[rst]]".format(
                res['http']['proxy']
            )
            return sdk_compat.build_error_result(parent, task_id, snippet, 'REGULAR', 'FAILED')
        else:
            snippet = "[[bad]]Task {} doesn't contain TEST_ENVIRONMENT_JSON_V2 resource".format(
                helpers.get_task_url(task_info['id'])
            )
            return sdk_compat.build_error_result(parent, task_id, snippet, 'REGULAR', 'FAILED')
    else:
        suite_info = helpers.get_suite_info(parent)
        if not suite_info or not suite_info.get('suite_hid'):
            raise Exception(
                "Child task {} failed with status {} (no expected_test_info was provided)".format(
                    task_id, task_info.get('status')
                )
            )

        error_type = "TIMEOUT" if task_info.get('status') == 'TIMEOUT' else "REGULAR"
        snippet = "[[bad]]Task {} failed with status [[imp]]{}[[rst]]: no test results can be obtained.\nTask info: {}".format(
            task_id,
            task_info.get('status'),
            helpers.html_to_markup(task_info['results']['info']),
        )
        return sdk_compat.build_error_result(parent, task_id, snippet, error_type, 'FAILED')
