import logging

from . import helpers
from .build import config as build_conf

import sandbox.common.platform as sb_platform
import sandbox.common.types.client as ctc
import sandbox.projects.common.constants as consts
import sandbox.sdk2 as sdk2

logger = logging.getLogger(__name__)

DEFAULT_CPU = 32
DEFAULT_CPU_MULTISLOT = 16
DEFAULT_DISK_USAGE = 100
DEFAULT_PLATFORM = "linux_ubuntu_16.04_xenial"
DEFAULT_RAM = 64
DEFAULT_RAM_DRIVE = 0

MAPSMOBI_TAG = "ya:exotic_platform_use_maps_mobile_hosts"
MAPSMOBI_ANDROID_TAG = "ya:maps_mobile_android"
# TODO(snowball): remove when DTCC-526 will be done
MAPSMOBI_OSX_BIG_SUR_TAG = "ya:exotic_platform_use_maps_mobile_osx_big_sur_hosts"
PORTOD_TAG = "sb:portod"
SEMIDIST_OWNERS = ["YT", "YT_ROBOT"]

MAPS_MOBILE_SPECIAL_CONTEXT = {
    "requested_arch": "osx",
    "binary_executor_release_type": "stable",
    consts.BUILD_SYSTEM_KEY: consts.YA_MAKE_FORCE_BUILD_SYSTEM,
    consts.USE_AAPI_FUSE: True,
    consts.USE_ARC_INSTEAD_OF_AAPI: True,
}

MAPS_MOBILE_YT_CACHE_CONTEXT = {
    consts.YA_YT_DIR: "//home/maps_mobile_build_cache",
    consts.YA_YT_PROXY: "hahn.yt.yandex.net",
    consts.YA_YT_PUT: True,
    consts.YA_YT_STORE: True,
    consts.YA_YT_STORE_CODEC: "zstd08_1",
    consts.YA_YT_TOKEN_VAULT_NAME: "robot-mapkit-ci-yt-token",
    consts.YA_YT_TOKEN_VAULT_OWNER: "robot-mapkit-ci",
    consts.YA_YT_STORE_REFRESH_ON_READ: True,
    consts.YA_YT_STORE_THREADS: 100,
}

MAPS_MOBILE_SPECIAL_REQUIREMENTS = {"cores": 4, "ram": 8, "disk_space": 10 * 1024}


def get_task(parent):
    import test.const as tconst

    tags = helpers.get_suite_tags(parent)

    if parent.Parameters.native_target:
        cls = LargeTestNativeWindows
    elif MAPSMOBI_ANDROID_TAG in tags:
        cls = LargeTestMapsAndroid
    elif MAPSMOBI_OSX_BIG_SUR_TAG in tags:
        cls = LargeTestMapsIosBigSur
    elif MAPSMOBI_TAG in tags:
        cls = LargeTestMapsIos
    elif tconst.YaTestTags.ExoticPlatform in tags:
        cls = LargeTestExoticPlatform
    else:
        cls = LargeTest
    return cls(parent)


class LargeTest(object):
    def __init__(self, parent):
        self.parent = parent
        self.context = {}
        self.requirements = {}
        self._max_retries = 0
        self._mutlslot_run = True

        self.init()
        self.init_debug()

    @property
    def task_type(self):
        return "YA_MAKE"

    def init(self):
        import devtools.ya.test.fat_test.config as large_test_config
        import test.const as tconst
        import yalibrary.yandex.distbuild.distbs_consts as dist_priority

        self.context = large_test_config.get_ya_make_task_default_options()
        # Inherit some options
        self.context.update(
            {
                "targets": self.parent.Parameters.targets,
                consts.ARCADIA_PATCH_KEY: self.parent.Parameters.arcadia_patch,
                consts.ARCADIA_URL_KEY: self.parent.Parameters.arcadia_url,
                consts.TEST_FILTERS: self.parent.Parameters.targets,
                consts.YA_YT_TOKEN_VAULT_OWNER: self.parent.owner,
            }
        )

        requirements = helpers.get_suite_requirements(self.parent)
        if requirements.get("container"):
            self.context[consts.SANDBOX_CONTAINER] = requirements["container"]

        tags = helpers.get_suite_tags(self.parent)
        sb_tag_groups, tag_params = helpers.parse_tags(tags)

        for group in sb_tag_groups:
            if set(group) & set(sb_platform.TAG_TO_PLATFORM.keys()):
                platform_tag = True
                break
        else:
            platform_tag = False

        if not platform_tag:
            self.context["requested_arch"] = DEFAULT_PLATFORM

        # Some sandbox tags can control task parameters
        resource_ttl = tag_params.get("ttl", "")
        if resource_ttl.isdigit():
            self.context[consts.BUILD_OUTPUT_TTL] = int(resource_ttl)

        # We need to specify suite type,
        # because target test might be a multimodule with several suites.
        suite_type = helpers.get_suite_type(self.parent)
        if suite_type:
            # XXX ya uses `pytest` as alias to run py2test and py3test suite types
            # For more info see https://st.yandex-team.ru/DEVTOOLS-7033
            if suite_type == "pytest":
                suite_type = "py2test"
            self.context[consts.TEST_TYPE_FILTER] = suite_type

        # Common configuration
        self.context.update(
            {
                consts.DIR_OUTPUTS: True,
                consts.REPORT_CONFIG_PATH: "autocheck/autocheck-config-fat.json",
                consts.TARGET_PLATFORM_FLAGS: build_conf.get_platform_flags(
                    toolchain=helpers.get_suite_toolchain(self.parent),
                    tags=tags,
                    target=self.parent.Parameters.targets,
                ),
            }
        )

        if self.parent.Parameters.native_target:
            self.context[consts.TESTS_REQUESTED] = False
            # Test result will be deduced in the YA_TEST_PARENT_2 task, depending on build and configure successfulness
            # For more info see https://st.yandex-team.ru/CI-1259
            self.context[consts.REPORT_TESTS_ONLY] = False

        if self.is_distbuild_allowed(tags):
            if self.is_force_distbuild(tags):
                self.context[consts.BUILD_SYSTEM_KEY] = consts.DISTBUILD_BUILD_SYSTEM
            else:
                self.context[consts.BUILD_SYSTEM_KEY] = consts.SEMI_DISTBUILD_BUILD_SYSTEM

            self.context[consts.RETRIABLE_DISPLACED_BUILD] = True
            self.context[consts.OUTPUT_ONLY_TESTS] = True
            # Add extra retries to overcome Distbuild"s displace error
            self.set_max_restarts(200)  # 5h at least

            revision = self.get_revision()
            if revision and self.parent.Parameters.is_precommit:
                self.context[consts.DISTBUILD_PRIORITY] = dist_priority.calc_dist_priority(
                    -1, revision, dist_priority.DISTBUILD_LARGE_TESTS_PRIORITY_DELTA
                )
        else:
            self.context[consts.OUTPUT_ONLY_TESTS] = False
            self.context[consts.BUILD_SYSTEM_KEY] = consts.YMAKE_BUILD_SYSTEM

        disable_arc = False
        # SANDBOX-8482 Porto socket avaliable ONLY in privileged porto containers
        if tconst.YaTestTags.Privileged in tags or PORTOD_TAG in tags:
            self.context.update(
                {
                    consts.PRIVILEGED: True,
                    # privileged containers mount arcadia cache in read only mode and cannot update it
                    consts.CHECKOUT: True,
                    consts.CLEAR_BUILD_KEY: True,
                }
            )

        if tconst.YaTestTags.NoFuse in tags:
            disable_arc = True

        try:
            sb_client_tags = ctc.Tag.Query(helpers.join_tags(sb_tag_groups)) if sb_tag_groups else None
        except:
            sb_client_tags = None

        if disable_arc:
            self.disable_multislot()
            sb_client_tags = sb_client_tags & ctc.Tag.HDD if sb_client_tags else ctc.Tag.HDD
            self.context.update(
                {
                    consts.USE_AAPI_FUSE: False,
                    consts.USE_ARC_INSTEAD_OF_AAPI: False,
                }
            )

            if self.parent.Parameters.arcadia_base:
                self.context[consts.ARCADIA_URL_KEY] = "arcadia:/arc/trunk/arcadia@{}".format(
                    self.parent.Parameters.arcadia_base
                )
            # TE backward compatibility - remote when TE is done
            else:
                parsed_url = sdk2.svn.Arcadia.parse_url(self.parent.Parameters.arcadia_url)
                if not parsed_url.revision:
                    raise Exception(
                        "Wrong arcadia url-format: {}, expected: {}".format(
                            self.parent.Parameters.arcadia_url, "arcadia:/arc/{svn_path}/arcadia@{revision}"
                        )
                    )

        if sb_client_tags:
            self.set_client_tags(str(sb_client_tags))

        if (
            tconst.YaTestTags.PerfTest in tags
            # WIP: see https://st.yandex-team.ru/DEVTOOLS-9405
            or not any('MULTISLOT' in x for x in sb_tag_groups)
        ):
            self.disable_multislot()

        if tconst.YaTestTags.YtRunner in tags:
            self.context.update(
                {
                    consts.RUN_TAGGED_TESTS_ON_YT: True,
                    consts.YA_YT_TOKEN_VAULT_NAME: "devtools-ya-test-yt-vanilla-execute",
                }
            )

        env_vars = {}
        if requirements.get("sb_vault"):
            vaults_vars = requirements["sb_vault"].split(",")
            for var in vaults_vars:
                name, val = var.split("=", 1)
                env_vars[name] = "$(vault:{})".format(val)

        if env_vars:
            self.context[consts.ENV_VARS] = " ".join(["{}={}".format(k, v) for k, v in env_vars.items()])

        if requirements.get("dns") == "dns64":
            self.context[consts.DNS64_REQUIRED] = True

        # Setup requirements
        disk_space_in_bytes = helpers.normalize_size_requirement(requirements.get("disk_usage", DEFAULT_DISK_USAGE))
        ram_in_bytes = helpers.normalize_size_requirement(requirements.get("ram", DEFAULT_RAM))
        ram_disk_in_bytes = helpers.normalize_size_requirement(requirements.get("ram_disk", DEFAULT_RAM_DRIVE))

        cores = tag_params.get("cores")
        if cores and cores.isdigit():
            cores = int(cores)

        # Sandbox automatically assign task to MULTISLOT host if it fits multislot's maximum capacity.
        # https://docs.yandex-team.ru/sandbox/agents#slots
        if self.multislot_allowed():
            if cores is None:
                cores = DEFAULT_CPU_MULTISLOT
        else:
            if cores is None or cores <= DEFAULT_CPU_MULTISLOT:
                # Use DEFAULT_CPU to prohibit running on multislot hosts
                cores = DEFAULT_CPU

        self.requirements.update(
            {
                "cores": cores,
                "disk_space": disk_space_in_bytes // 1024**2,  # in MB
                "ram": ram_in_bytes // 1024**2,  # in MB
            }
        )

        if ram_disk_in_bytes:
            self.context[consts.RAM_DRIVE_SIZE] = ram_disk_in_bytes / 1024**2  # in MB

        # XXX Hacks section. It would be nice to get rid of these.
        # Use custom timeouts for users using own quote.
        # For more info see DEVTOOLSSUPPORT-3435
        if not self.is_force_distbuild(tags):
            if self.parent.Parameters.targets.startswith("yt/"):
                ya_timeout = 6 * 3600
                self.context.update(
                    {
                        consts.YA_TIMEOUT: ya_timeout,
                        "kill_timeout": ya_timeout + 3600,
                    }
                )
            else:
                ya_timeout = 4 * 3600
                self.context.update(
                    {
                        consts.YA_TIMEOUT: ya_timeout,
                        "kill_timeout": ya_timeout + 600,
                    }
                )

    def init_debug(self):
        if self.parent.Parameters.debug_all_test_tag:
            self.context[consts.TEST_TAG] = "ya:__any_tag"

    def is_distbuild_allowed(self, tags):
        return self.is_force_distbuild(tags) or self.parent.owner in SEMIDIST_OWNERS

    def is_force_distbuild(self, tags):
        return "ya:force_distbuild" in tags

    def set_max_restarts(self, val):
        self._max_retries = val
        self.context["max_restarts"] = val

    def disable_multislot(self):
        self._mutlslot_run = False

    def multislot_allowed(self):
        return self._mutlslot_run is True

    @property
    def max_restarts(self):
        return self._max_retries

    def get_revision(self):
        parsed_url = sdk2.svn.Arcadia.parse_url(self.parent.Parameters.arcadia_url)
        if parsed_url.revision:
            if parsed_url.revision.isdigit():
                return int(parsed_url.revision)

        if self.parent.Parameters.autocheck_revision:
            return int(self.parent.Parameters.autocheck_revision)
        elif parsed_url.revision:
            raise Exception(
                "Wrong arcadia url-format: {}, expected: {}".format(
                    self.parent.Parameters.arcadia_url, "arcadia:/arc/{svn_path}/arcadia@{revision}"
                )
            )
        return None

    def set_client_tags(self, tags):
        self.context[consts.SANDBOX_TAGS] = tags
        self.requirements["client_tags"] = tags


class LargeTestNativeWindows(LargeTest):
    @property
    def task_type(self):
        return "YA_MAKE_2"

    def init(self):
        super(LargeTestNativeWindows, self).init()
        self.set_client_tags("WINDOWS")
        self.context.update(
            {
                "binary_executor_release_type": "stable",
                "env_vars": "ENABLE_DEVTOOLSSUPPORT_13736=yes",
            }
        )
        # XXX hacks
        # https://st.yandex-team.ru/DEVTOOLSSUPPORT-13568
        helpers.switch_arcadia_url_to_arc(self.context)

        self.requirements.update(
            {
                "cores": 4,
                "disk_space": 40 * 1024,
                "ram": 8 * 1024,
            }
        )


class LargeTestMapsAndroid(LargeTest):
    def init(self):
        super(LargeTestMapsAndroid, self).init()
        self.context.update(MAPS_MOBILE_YT_CACHE_CONTEXT)


class LargeTestMapsIos(LargeTest):
    @property
    def task_type(self):
        return "YA_MAKE_MAPS_MOBILE_OSX"

    def init(self):
        super(LargeTestMapsIos, self).init()
        self.context.update(MAPS_MOBILE_SPECIAL_CONTEXT)
        self.context.update(MAPS_MOBILE_YT_CACHE_CONTEXT)
        self.requirements.update(MAPS_MOBILE_SPECIAL_REQUIREMENTS)
        self.set_client_tags("CUSTOM_MAPS_MOBILE_DARWIN")
        # XXX hacks
        # https://st.yandex-team.ru/DEVTOOLSSUPPORT-11983
        helpers.switch_arcadia_url_to_arc(self.context)


class LargeTestNative(LargeTest):
    @property
    def task_type(self):
        return "YA_MAKE_2"

    def init(self):
        super(LargeTestNative, self).init()
        tags = helpers.get_suite_tags(self.parent)
        sb_tag_groups, tag_params = helpers.parse_tags(tags)
        self.set_client_tags(helpers.join_tags(sb_tag_groups))
        xcode_version = tag_params.get("xcode", "")
        if xcode_version != "":
            self.context[consts.XCODE_VERSION] = xcode_version
        log_ttl = tag_params.get("ttl", "")
        if log_ttl.isdigit():
            self.context[consts.BUILD_LOGS_TTL] = log_ttl
        force_sandbox_tags = tag_params.get("force_sandbox_tags", "false")
        if force_sandbox_tags == "true":
            self.context["force_sandbox_tags"] = True

        # XXX hacks
        # https://st.yandex-team.ru/DEVTOOLSSUPPORT-13568
        helpers.switch_arcadia_url_to_arc(self.context)


class LargeTestMapsIosBigSur(LargeTestNative):
    def init(self):
        super(LargeTestMapsIosBigSur, self).init()
        self.context.update(MAPS_MOBILE_SPECIAL_CONTEXT)
        self.context.update(MAPS_MOBILE_YT_CACHE_CONTEXT)
        self.requirements.update(MAPS_MOBILE_SPECIAL_REQUIREMENTS)


class LargeTestExoticPlatform(LargeTest):
    @property
    def task_type(self):
        return "YA_MAKE_TEST_WITH_BUILD_HEATER"

    def init(self):
        super(LargeTestExoticPlatform, self).init()
        self.context.update(
            {
                # Options for build task
                consts.ALLOW_AAPI_FALLBACK: True,
                consts.BUILD_SYSTEM_KEY: consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                consts.USE_AAPI_FUSE: True,
                # Options for test task
                "testing_platform": self.get_test_host_filter(),
                # Drop ram drive for darwin - it's not supported
                consts.RAM_DRIVE_SIZE: 0,
            }
        )
        self.requirements.update(
            {
                "cores": 2,
                "disk_space": 10 * 1024,
                "ram": 4 * 1024,
            }
        )
        self.set_client_tags("GENERIC")

    def get_test_host_filter(self):
        toolchain = helpers.get_suite_toolchain(self.parent)
        if "-darwin-" in toolchain or "-ios-" in toolchain:
            return "osx"
        return "linux"
