# -*- coding: utf-8 -*-
import logging
import subprocess

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task

MB = 1024 * 1024


class YtcacheDataGcBin(sdk2.Resource):
    releasable = True
    any_arch = False
    executable = True


class YtCacheCleaner(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Задача для очистки YT-кеша сборки от устаревших данных.

    Ожидается, что таблицы кеша созданы в новом формате:
        - таблица metadata содержит поля access_time и data_size
        - таблица data содержит поле create_time
        - обе таблицы не имеют атрибута max_data_ttl, т.е. не очищаются средствами YT

    Очистка делается двумя дополняющими друг друга способами:
        1. Регулярное удаление данных по сроку хранения или размеру кеша.
           Эта операция относительная лёгкая и может запускаться с любой разумной периодичностью или вручную.
        2. Удаление из таблицы data строк, на которые нет ссылок из таблицы metadata.
           При нормальной работе кеша таких строк образовываться не должно, тем не менее, при различного рода сбоях
           такая ситуацияю возможна.
           ВНИМАНИЕ: Это очень тяжёлая операция, поэтому рекомендуется её выполнять раз в день или реже.
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        yt_proxy = sdk2.parameters.String('YtProxy', required=True)
        yt_dir = sdk2.parameters.String('YtDir', required=True)
        readonly = sdk2.parameters.Bool('Read only', default=False)
        yt_token = sdk2.parameters.YavSecret("YT_TOKEN YAV secret identifier", required=True)

        with sdk2.parameters.RadioGroup("Cleaning mode") as mode:
            mode.values["regular"] = mode.Value("Regular", default=True)
            mode.values["forced_compaction"] = mode.Value("Forced compaction")
            mode.values["gc"] = mode.Value("Data table garbage collector")
            mode.values["stat"] = mode.Value("Merge chunks in stat table")

        with mode.value["regular"]:
            max_cache_size_mb = sdk2.parameters.Integer('Max Cache Size in Mb')
            ttl = sdk2.parameters.Integer('TTL in hours', default=24)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "YT_CACHE_CLEANER", "released": self.Parameters.binary_executor_release_type},
            "owner": "YATOOL",
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        yt_token = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
        if self.Parameters.mode == "regular":
            self._regular_cleaning(yt_token)
        elif self.Parameters.mode == "forced_compaction":
            self._start_forced_compaction(yt_token)
        elif self.Parameters.mode == "gc":
            self._gc_data(yt_token)
        elif self.Parameters.mode == "stat":
            self._merge_stat_table(yt_token)
        else:
            raise Exception("Unknown cleaning mode: " + self.Parameters.mode)

    def _regular_cleaning(self, token):
        from yalibrary.store.yt_store import yt_store

        cache = yt_store.YtStore(
            self.Parameters.yt_proxy,
            self.Parameters.yt_dir,
            None,
            token=token,
            readonly=self.Parameters.readonly,
            max_cache_size=self.Parameters.max_cache_size_mb * MB,
            ttl=self.Parameters.ttl,
        )

        status_before_clean = cache.get_status()
        counters = cache.strip()
        if counters:
            logging.info('Deleted: meta rows:%d, data rows:%d, net data size:%d',
                         counters['meta_rows'], counters['data_rows'], counters['data_size'])
        status_after_clean = cache.get_status()

        stat = {
            'before_clean': status_before_clean,
            'after_clean': status_after_clean,
        }
        cache.put_stat('cleaner', stat)

    def _start_forced_compaction(self, token):
        from yalibrary.store.yt_store import yt_store
        cache = yt_store.YtStore(
            self.Parameters.yt_proxy,
            self.Parameters.yt_dir,
            None,
            token=token,
        )
        cache.start_forced_compaction()

    def _merge_stat_table(self, token):
        from yalibrary.store.yt_store import yt_store
        cache = yt_store.YtStore(
            self.Parameters.yt_proxy,
            self.Parameters.yt_dir,
            None,
            token=token,
        )
        cache.merge_stat()

    def _gc_data(self, token):
        resource = YtcacheDataGcBin.find(attrs={"released": "stable"}).first()
        data = sdk2.ResourceData(resource)
        binary_path = data.path
        args = [
            str(binary_path),
            "--yt-proxy", self.Parameters.yt_proxy,
            "--yt-dir", self.Parameters.yt_dir.rstrip('/'),
        ]
        if self.Parameters.readonly:
            args += ["--dry-run"]
        with sdk2.helpers.ProcessLog(self, logger="ytcache-data-gc") as pl:
            env = {
                "YT_TOKEN": token,
                "LOG_LEVEL": "debug",
            }
            subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stderr, env=env)
