import logging
import os
import time
import json
from collections import namedtuple
import re

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.build.YaMake2 import YaMake2, YaMakeParameters
from sandbox.sdk2.vcs.svn import Arcadia
import sandbox.common as common
import sandbox.common.types.resource as ctr
import sandbox.projects.common.constants as consts


DESIGNATING_ATTRIBUTE_NAME = "arcadia_yt_heater"
NAMESPACE_PATTERN = re.compile(r'"namespace"\s*:\s*"yt_store"')
EVENT_PATTERN = re.compile(r'"event"\s*:\s*"stats"')
DEFAULT_YT_PROXY = 'hahn'
DEFAULT_YT_DIR = '//home/devtools/cache'
DEFAULT_VAULT_NAME = 'YT_STORE_TOKEN'


class PutHeaterStatException(Exception):
    pass


class YtHeater(YaMake2):
    class Requirements(task_env.BuildLinuxRequirements):
        pass

    class Parameters(YaMakeParameters):
        with sdk2.parameters.Group('Internal parameters') as internal_parameters:
            allow_revision = sdk2.parameters.Bool('Allow revision heating', default=False)
            bin_params = binary_task.LastBinaryReleaseParameters()
            heater_name = sdk2.parameters.String('Heater name for statistics', default=None)

    @property
    def binary_executor_query(self):
        return {
            'attrs': {DESIGNATING_ATTRIBUTE_NAME: True, 'released': self.Parameters.binary_executor_release_type},
            'owner': 'YATOOL',
            'state': [ctr.State.READY],
        }

    class Context(sdk2.Context):
        create_results_resource = False

    def get_dist_priority(self):
        logging.debug("YT_HEATER priority: %s", self.Context.normalized_dist_priority)
        return self.Context.normalized_dist_priority

    def get_heater_mode(self):
        return True

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        start_time = time.time()
        self.__check_url()

        token = self._get_token()
        yt_store = self._get_yt_store(token)
        size_checher = YtCacheSizeChecker(yt_store, getattr(self.Parameters, consts.YA_YT_MAX_CACHE_SIZE, None))
        arc_api = ArcCommitMicroApi(token)

        size_checher.check()
        YaMake2.on_execute(self)
        size_checher.check()  # Check size again to catch cache overflow during heating

        try:
            self._put_heater_stat(yt_store, arc_api, start_time)
        except PutHeaterStatException as e:
            self.set_info("Cannot put heater stat: {}".format(str(e)))
            logging.exception("Cannot put heater stat")

    def pre_build(self, source_dir):
        self.__set_distbuild_priority()
        YaMake2.pre_build(self, source_dir)

    def get_additional_build_def_flags(self):
        flags = YaMake2.get_additional_build_def_flags(self)
        # Set the same value as in local builds
        flags['SANDBOX_TASK_ID'] = 0
        return flags

    def __set_distbuild_priority(self):
        from yalibrary.yandex.distbuild import distbs_consts as dp

        default_priority = getattr(self.Parameters, consts.DISTBUILD_PRIORITY, None)
        try:
            self.Context.frozen_arcadia_from_url = Arcadia.freeze_url_revision(self.Parameters.checkout_arcadia_from_url)
            parsed_url = Arcadia.parse_url(self.Context.frozen_arcadia_from_url)
            if not parsed_url.trunk or not parsed_url.revision:
                logging.debug("YT_HEATER default priority due to url(%s), revision(%s)", parsed_url, parsed_url.revision)
                self.Context.normalized_dist_priority = default_priority
                return

            self.Context.svn_revision = parsed_url.revision
            revision = int(self.Context.svn_revision)
        except Exception as exc:
            logging.error("Exception caught while setting priority: %s", exc)
            self.Context.normalized_dist_priority = default_priority
            return

        self.Context.normalized_dist_priority = dp.calc_dist_priority(-1, revision)
        if self.owner in ['YATOOL', 'REVIEW-CHECK', 'REVIEW-CHECK-FAT', 'AUTOCHECK-FAT', 'YA_YT_CACHE']:
            self._append_to_gsid('SB:YT_HEATER_FOR_ARCADIA')
        self._append_to_gsid(self._get_parameters_marker())
        self._append_to_gsid("YT:svn_revision={}".format(self.Context.svn_revision))
        os.environ["GSID"] = str(self.Context.__GSID)
        logging.debug("YT_HEATER: distbuild_priority=%s, task owner=%s, _GSID=%s", self.Context.normalized_dist_priority, self.owner, self.Context.__GSID)

    def __check_url(self):
        parsed_url = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
        if not parsed_url.trunk or (parsed_url.revision and not self.Parameters.allow_revision):
            raise common.errors.TaskFailure("YT_HEATER should not be used with non-trunk branch/tag or with specific revision")
        logging.debug("YT_HEATER: heating for url: %s", parsed_url)

    def _append_to_gsid(self, name):
        if name not in self.Context.__GSID:
            self.Context.__GSID += ' ' + name

    def _get_parameters_marker(self):
        marker = "YT"
        marker += ":{}".format(self.Parameters.build_type)

        target_flags = getattr(self.Parameters, consts.TARGET_PLATFORM_FLAGS, '')
        if 'DARWIN' in target_flags:
            marker += ":DARWIN"
        elif "WINDOWS" in target_flags:
            marker += ":WINDOWS"
        else:
            marker += ":LINUX"

        flags = self.Parameters.definition_flags.split(' ')
        index = "0"
        count = "1"
        common = False
        for f in flags:
            if f.startswith('-DRECURSE_PARTITIONS_COUNT='):
                count = f.split('=')[1]
            elif f.startswith('-DRECURSE_PARTITION_INDEX='):
                index = f.split('=')[1]
            elif f in ('-DCOMMON_PROJECTS', '-DCOMMON_PROJECTS=yes'):
                common = True

        marker += ":INDEX={}:COUNT={}".format(index, count)
        if common:
            marker += ":COMMON_PROJECTS"

        if getattr(self.Parameters, consts.MUSL, False):
            marker += ":musl"

        return marker

    def _get_token(self):
        token_owner = getattr(self.Parameters, consts.YA_YT_TOKEN_VAULT_OWNER) or self.owner
        token_name = getattr(self.Parameters, consts.YA_YT_TOKEN_VAULT_NAME) or DEFAULT_VAULT_NAME
        return sdk2.Vault.data(token_owner, token_name)

    def _get_yt_store(self, token):
        from yalibrary.store.yt_store import yt_store
        yt_proxy = getattr(self.Parameters, consts.YA_YT_PROXY) or DEFAULT_YT_PROXY
        yt_dir = getattr(self.Parameters, consts.YA_YT_DIR) or DEFAULT_YT_DIR
        return yt_store.YtStore(
            yt_proxy,
            yt_dir,
            None,
            token=token,
        )

    def _put_heater_stat(self, yt_store, arc_api, start_time):
        if not self.Parameters.heater_name:
            raise PutHeaterStatException('heater_name parameter is empty')
        evlogs = list(self.path().glob('*evlog*.json'))
        if not evlogs:
            raise PutHeaterStatException('evlog not found')

        def search_stat_line(evlogs):
            for evlog in evlogs:
                with evlog.open() as f:
                    for line in f:
                        if NAMESPACE_PATTERN.search(line) and EVENT_PATTERN.search(line):
                            return line

        stat_line = search_stat_line(evlogs)
        if not stat_line:
            raise PutHeaterStatException('yt_store statistics not found in evlog')

        evlog_data = json.loads(stat_line)
        commit = arc_api.get_commit(self.Context.svn_revision)
        stat = {
            'sandbox_task_id': self.id,
            'name': self.Parameters.heater_name,
            'svn_revision': commit.svn_revision,
            'arc_oid': commit.oid,
            'commit_timestamp': commit.timestamp,
            'exec_duration': int(time.time() - start_time),
            'stats': evlog_data['value'],
        }
        yt_store.put_stat('heater', stat)


class YtCacheSizeChecker(object):
    def __init__(self, store, max_cache_size):
        self._max_cache_size = max_cache_size
        self._store = store
        if not self._max_cache_size:
            logging.debug('YT_HEATER: Don\'t check yt cache size because max_cache_size is omitted or equals to zero')

    def check(self):
        if not self._max_cache_size:
            return
        size = self._store.get_used_size()
        if size > self._max_cache_size:
            raise common.errors.TaskFailure('YT store size limit exceeded ({} > {})'.format(size, self._max_cache_size))


class ArcCommitMicroApi(object):
    ARC_API_URL = 'api.arc-vcs.yandex-team.ru:6734'

    Commit = namedtuple('Commit', 'svn_revision oid timestamp')

    def __init__(self, token):
        import grpc
        from arc.api.public.repo_pb2_grpc import CommitServiceStub
        token_credentials = grpc.access_token_call_credentials(token)
        channel_credentials = grpc.composite_channel_credentials(grpc.ssl_channel_credentials(), token_credentials)
        channel = grpc.secure_channel(self.ARC_API_URL, channel_credentials)
        self._service = CommitServiceStub(channel)

    def get_commit(self, svn_revision):
        from arc.api.public.repo_pb2 import GetCommitRequest
        req = GetCommitRequest(Revision="r" + svn_revision)
        resp = self._service.GetCommit(req)
        return self.Commit(resp.Commit.SvnRevision, resp.Commit.Oid, resp.Commit.Timestamp.ToMicroseconds())
