# -*- coding: utf-8 -*-
import logging
from collections import namedtuple

from sandbox import sdk2
from sandbox.projects.common import binary_task
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt


class YtHeaterLauncher(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('ArcadiaUrl', required=True)
        template_alias = sdk2.parameters.String('Template alias', required=True)
        branch_name = sdk2.parameters.String('Branch name', required=True)
        heater_parameters = sdk2.parameters.JSON('Heater parameter list', default_value='[]', required=True)
        arc_token = sdk2.parameters.YavSecret('Arc token secret', required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            'attrs': {'task_type': 'YT_HEATER_LAUNCHER_BINARY', 'released': self.Parameters.binary_executor_release_type},
            'owner': 'YATOOL',
            'state': [ctr.State.READY]
        }

    def on_execute(self):
        arc_token = self.Parameters.arc_token.data()[self.Parameters.arc_token.default_key]
        arc_api = ArcBranchMicroApi(arc_token)

        with self.memoize_stage.get_ref:
            ref = arc_api.get_trunk_ref()
            self.Context.svn_revision = ref.svn_revision
            self.Context.arc_oid = ref.oid

        with self.memoize_stage.create_children:
            checkout_arcadia_from_url = '{}@{}'.format(self.Parameters.arcadia_url, self.Context.svn_revision)

            # Create tasks in draft state
            heater_task_ids = []
            for heater_params in self.Parameters.heater_parameters:
                task = self.server.task({
                    'children': True,
                    'template_alias': self.Parameters.template_alias,
                })
                task_id = task['id']
                update = {
                    'custom_fields': [
                        {'name': 'checkout_arcadia_from_url', 'value': checkout_arcadia_from_url},
                        {'name': 'allow_revision', 'value': True},
                    ],
                }
                for name, value in heater_params.items():
                    if name in ('description', ):
                        # Task attributes (not input_parameters)
                        update[name] = value
                    else:
                        update['custom_fields'].append({'name': name, 'value': value})

                self.server.task[task_id].update(update)
                heater_task_ids.append(task_id)

            # Start created tasks
            for task_id in heater_task_ids:
                r = self.server.batch.tasks.start.update(task_id)[0]
                if r['status'] == 'ERROR':
                    raise Exception('Cannot start task {}: {}'.format(task_id, r['message']))

            self.Context.heater_task_ids = heater_task_ids
            raise sdk2.WaitTask(heater_task_ids, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK], wait_all=True)

        for task_id in self.Context.heater_task_ids:
            t = self.server.task[task_id].read()
            status = t['status']
            if status != 'SUCCESS':
                raise Exception('Task {} failed'.format(task_id))

        logging.info('Set %s to svn revision=%d (oid=%s)', self.Parameters.branch_name, self.Context.svn_revision, self.Context.arc_oid)
        arc_api.set_branch_ref(self.Parameters.branch_name, self.Context.arc_oid)


class ArcBranchMicroApi(object):
    ARC_API_URL = 'api.arc-vcs.yandex-team.ru:6734'
    TRUNK_NAME = 'trunk'

    Ref = namedtuple('Ref', 'svn_revision oid')

    def __init__(self, token):
        import grpc
        from arc.api.public.repo_pb2_grpc import BranchServiceStub
        token_credentials = grpc.access_token_call_credentials(token)
        channel_credentials = grpc.composite_channel_credentials(grpc.ssl_channel_credentials(), token_credentials)
        channel = grpc.secure_channel(self.ARC_API_URL, channel_credentials)
        self._service = BranchServiceStub(channel)

    def get_trunk_ref(self):
        from arc.api.public.repo_pb2 import ListRefsRequest
        req = ListRefsRequest(Inconsistent=True, PrefixFilter=self.TRUNK_NAME)
        resp = self._service.ListRefs(req)
        for batch in resp:
            for ref in batch.Refs:
                if ref.Name == self.TRUNK_NAME:
                    return self.Ref(ref.Commit.SvnRevision, ref.Commit.Oid)

    def set_branch_ref(self, branch_name, oid):
        from arc.api.public.repo_pb2 import SetRefRequest
        req = SetRefRequest(Branch=branch_name, CommitOid=oid, Force=True)
        self._service.SetRef(req)
