# coding: utf-8

import json
import logging

from devtools.ya.test import facility
from test import const
from test.test_types import common as test_types_common

logger = logging.getLogger(__file__)


def gen_suite(suite_error=None, **kwargs):
    suite = test_types_common.PerformedTestSuite(**kwargs)
    if suite_error:
        suite.add_suite_error(suite_error)
    # TODO prettyboy@ remove after switch
    if hasattr(suite, 'register_chunk'):
        suite.register_chunk()
    return suite


def get_test_name(entry):
    return "{}::{}".format(entry.get('name', 'name-unknown'), entry.get('subtest_name', 'subtest_name-unknown'))


def get_test_status(entry):
    status = entry.get('status', 'REGULAR')
    if status == 'SKIPPED':
        return const.Status.SKIPPED
    elif status == 'DISCOVERED':
        return const.Status.DESELECTED
    elif status == 'OK':
        return const.Status.GOOD
    elif status == 'NOT_LAUNCHED':
        return const.Status.NOT_LAUNCHED
    elif status == 'FAILED':
        return {
            "BROKEN_DEPS": const.Status.FAIL,
            "FLAKY": const.Status.FLAKY,
            "INTERNAL": const.Status.INTERNAL,
            "REGULAR": const.Status.FAIL,
            "TIMEOUT": const.Status.TIMEOUT,
            "XFAILED": const.Status.XFAIL,
            "XPASSED": const.Status.XPASS,
        }.get(entry['error_type'], const.Status.FAIL)

    raise AssertionError('Unknown entry status: {}'.format(status))


def get_links(entry):
    return {k: v[0] for k, v in entry.get('links', {}).items()}


def add_test(suite, entry):
    testcase = facility.TestCase(
        name=get_test_name(entry),
        status=get_test_status(entry),
        comment=entry.get('rich-snippet', ''),
        elapsed=entry.get('duration', 0.0),
        logs=get_links(entry),
        path=entry.get('path'),
        tags=entry.get('tags'),
    )
    # TODO prettyboy@ remove after switch
    if hasattr(suite, 'chunk'):
        suite.chunk.tests.append(testcase)
    else:
        suite.tests.append(testcase)


def parse_from_file(filename):
    try:
        with open(filename) as afile:
            data = json.load(afile)
    except OSError:
        logger.exception('Failed to load %s', filename)
        data = {}

    return parse(data.get('results', []))


def parse(results):
    if not results:
        return [
            gen_suite(
                suite_error="Results.json doesn't contain any information, looks like 'ya make' failed or YA_MAKE task was misconfigured",
                name='build',
                project_path='unknown',
                size='unknown',
            )
        ]

    suite_map = {}

    for entry in results:
        etype = entry.get('type', 'type-unknown')
        path = entry.get('path', 'path-unknown')
        toolchain = entry.get('toolchain', 'toolchain-unknown')

        key = '{}-{}-{}'.format(path, etype, toolchain)

        if entry.get('error_type'):
            error_msg = entry.get('rich-snippet', "Error doesn't contain rich-snippet")
        else:
            error_msg = None

        if etype == 'test':
            if key not in suite_map:
                suite_map[key] = gen_suite(
                    name=entry.get('name', 'name-unknown'),
                    project_path=path,
                    size=entry.get('size', 'size-unknown'),
                    target_platform_descriptor=toolchain,
                    suite_type=etype,
                    tags=entry.get('tags', []),
                    suite_ci_type=etype,
                    uid=entry.get('uid', 'uid-unknown'),
                )

            suite = suite_map[key]

            if entry.get('suite'):
                # Set actual suite type
                suite._suite_type = entry.get('name', 'name-unknown')
                if error_msg:
                    suite.add_suite_error(error_msg)
                suite.logs = get_links(entry)
            else:
                add_test(suite, entry)

        elif error_msg:
            suite_map[key] = gen_suite(
                suite_error="[[bad]]Error:[[rst]]{}".format(error_msg) if error_msg else None,
                name=etype,
                project_path=path,
                target_platform_descriptor=toolchain,
                suite_type=etype,
                suite_ci_type=etype,
                uid=entry.get('uid', 'uid-unknown'),
            )

    # 'configure' and 'build' before 'test'
    return sorted(suite_map.values(), key=lambda x: x.get_ci_type_name())
