# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, unicode_literals

import os
import shutil
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.gnupg import GpgKey2


class GpgKey3(GpgKey2):
    """Еще одна оберточка, деарморящая ключики и явно импортящая ключи."""

    PRIV_KEY_NAME_TMPL = "secring.{}"
    PUBL_KEY_NAME_TMPL = "pubring.{}"
    ARMORED_FILEEXT = "asc"
    DEARMORED_FILEEXT = "asc.gpg"
    NORMAL_FILEEXT = "gpg"

    def _gnupg_file(self, filetype, fileext):
        return os.path.join(self.gnupg_dir, filetype.format(fileext))

    @staticmethod
    def _import_keys(*keys):
        for key in keys:
            with sdk2.helpers.ProcessLog(logger=logging.getLogger("import_gpg_{}".format(key))) as pl:
                subprocess.Popen(["gpg", "--import", key], stdout=pl.stdout, stderr=pl.stderr).wait()

    @staticmethod
    def _dearmor_keys(*keys):
        for armored_key in keys:
            with sdk2.helpers.ProcessLog(logger=logging.getLogger("dearmor_gpg_{}".format(armored_key))) as pl:
                retcode = subprocess.Popen(["gpg", "--dearmor", armored_key], stdout=pl.stdout, stderr=pl.stderr).wait()
            if retcode:
                raise SandboxTaskFailureError("Cannot dearmor gpg keys (see task log for details)")

    def __enter__(self, do_key_import=True, do_key_dearmoring=True):
        if os.path.exists(self.gnupg_dir):
            shutil.move(self.gnupg_dir, self.TEMPGNUPG_DIR)
        try:
            os.mkdir(self.gnupg_dir)
        except OSError:
            pass

        if do_key_dearmoring:
            for key_type, keyvar in ((self.PUBL_KEY_NAME_TMPL, self._public_key),
                                     (self.PRIV_KEY_NAME_TMPL, self._secret_key)):
                armored = self._gnupg_file(key_type, self.ARMORED_FILEEXT)
                dearmored = self._gnupg_file(key_type, self.DEARMORED_FILEEXT)
                with open(armored, "wb+") as keyf:
                    keyf.write(self.get_vault_data(keyvar))
                self._dearmor_keys(armored)
                shutil.move(dearmored, self._gnupg_file(key_type, self.NORMAL_FILEEXT))
        else:
            private_key_path = self._gnupg_file(self.PRIV_KEY_NAME_TMPL, self.NORMAL_FILEEXT)
            public_key_path = self._gnupg_file(self.PUBL_KEY_NAME_TMPL, self.NORMAL_FILEEXT)
            open(public_key_path, "wb+").write(self.get_vault_data(self._public_key))
            open(private_key_path, "wb+").write(self.get_vault_data(self._secret_key))
        if do_key_import:
            self._import_keys(
                self._gnupg_file(self.PRIV_KEY_NAME_TMPL, self.NORMAL_FILEEXT),
                self._gnupg_file(self.PUBL_KEY_NAME_TMPL, self.NORMAL_FILEEXT)
            )
        logging.info("Contents of gnupg dir: {}".format(os.listdir(self.gnupg_dir)))
        return self
