# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, unicode_literals

import datetime
import hashlib
import logging
import os
import random
import shutil
import struct
import sys
import tarfile

from Crypto.Cipher import AES

from sandbox import sdk2
# Использование сандбоксовского модуля небезопасно, так как все аргументы в итоге сохраняются в логах
# (а палить их нам ни к чему).
import subprocess32 as subprocess

from sandbox.projects.di.resources import PydiMySQLBackup


class PydiMysqlBackup(sdk2.Task):
    """
    Бэкапит продакшн-Mysql интерфейса дистрибуции.
    """
    class Requirements(sdk2.Requirements):
        disk_space = 100 * 1024  # 10 Gb => 25 for db_backup + 25 for tmp storage + 50 for packing

    class Parameters(sdk2.Parameters):
        token_vault_onwner = sdk2.parameters.String(
            "Vault owner for DI MySQL password key",
            default="DI"
        )
        token_vault_password_key = sdk2.parameters.String(
            "Vault key name for DI MySQL password",
            default="di_prod_mysql_password"
        )
        database_user = sdk2.parameters.String("Database user", default="partner")
        databases_to_dump = sdk2.parameters.List("List of databases to dump", default=["distribution2", "kladr"])
        mysql_host = sdk2.parameters.String("DI MySQL host", default="pydi-mysql01e.haze.yandex.net")
        mysql_port = sdk2.parameters.Integer("DI MySQL port", default=3478)

    def _encrypt_backup(self, backup_file):
        """
        Пароль на бэкап - sha256 от пароля базы.
        """
        tmp_loc = str(self.path("tmp_backup.sql"))
        shutil.move(backup_file, tmp_loc)
        key = hashlib.sha256(sdk2.Vault.data(
            self.Parameters.token_vault_onwner,
            self.Parameters.token_vault_password_key,
        )).digest()

        # https://eli.thegreenplace.net/2010/06/25/aes-encryption-of-files-in-python-with-pycrypto
        iv = "".join(chr(random.randint(0, 0xFF)) for i in range(16))
        encryptor = AES.new(key, AES.MODE_CBC, iv)
        filesize = os.path.getsize(tmp_loc)

        with open(tmp_loc, "rb") as infile:
            with open(backup_file, "w+b") as outfile:
                outfile.write(struct.pack("<Q", filesize))
                outfile.write(iv)

                while True:
                    chunk = infile.read(64*1024)
                    if len(chunk) == 0:
                        break
                    elif len(chunk) % 16 != 0:
                        chunk += " " * (16 - len(chunk) % 16)

                    outfile.write(encryptor.encrypt(chunk))

    def _fetch_from_mysql(self, database):
        backup_filepath = str(self.path("{}_backup.sql".format(database)))
        backup_file = open(backup_filepath, "w+")
        subprocess.Popen(
            [
                "mysqldump", "--compatible=postgresql", "--default-character-set=utf8",
                "-u", self.Parameters.database_user, "-h", self.Parameters.mysql_host,
                "-P", str(self.Parameters.mysql_port),
                "--password={}".format(
                    sdk2.Vault.data(
                        self.Parameters.token_vault_onwner,
                        self.Parameters.token_vault_password_key,
                    ),
                ),
                database,
            ],
            stdout=backup_file,
            stderr=sys.stderr,
        ).wait()
        backup_file.close()
        self._encrypt_backup(backup_filepath)
        return backup_filepath

    def on_execute(self):
        tarfile_path = str(self.path("di_backup.tar.gz"))
        with tarfile.open(tarfile_path, "w:gz") as backup_archive:
            for database in self.Parameters.databases_to_dump:
                logging.info("Fetching: {}".format(database))
                backup_file = self._fetch_from_mysql(database)
                backup_archive.add(backup_file, os.path.basename(backup_file))
        sdk2.ResourceData(
            PydiMySQLBackup(self, "Backed up at {}. Database host {}".format(
                datetime.datetime.now(), self.Parameters.mysql_host
            ), path=tarfile_path, ttl="inf")
        ).ready()
