# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.sdk2 import yav
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.di.resources import DiJobs
from sandbox.projects.di.settings import SANDBOX_VAULT

JOB_TYPES = (
    ("Расчет активаций", "activations"),
    ("Отправка мобильных активаций в БК", "mobile_activations"),
    ("Обновление срезов Директа", "direct_revenue_share"),
    ("Обновление срезов Поиска", "searches"),
    ("Расчет установок", "installs"),
)


class DiJob(sdk2.Task):
    """Регулярные задачи Интерфейса Дистрибуции."""

    class Parameters(sdk2.Parameters):

        job_type = sdk2.parameters.String(
            "Job type",
            required=True,
            choices=JOB_TYPES,
        )

        job_binary = sdk2.parameters.LastResource(
            "Job binary",
            resource_type=DiJobs,
            owner="DI",
            attrs={"released": "stable"},
        )

        force = sdk2.parameters.Bool(
            "Force",
            required=True,
            default=False,
        )

    class Requirements(sdk2.Requirements):
        disk_space = 2 * 1024  # 2 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        secret = yav.Secret(SANDBOX_VAULT).data()

        if self.Parameters.job_binary:
            di_jobs_resource = self.Parameters.job_binary
        else:
            di_jobs_resource = DiJobs.find(
                owner="DI",
                state=State.READY,
                attrs={"released": "stable"}
            ).first()

        logging.info("Using di-jobs from resource = %s", di_jobs_resource.id)

        args = [str(sdk2.ResourceData(di_jobs_resource).path), self.Parameters.job_type]
        if self.Parameters.force:
            args.append("-f")

        run_process(
            args,
            log_prefix="job",
            environment={
                "YT_TOKEN": secret["YT_TOKEN"],
                "YQL_TOKEN": secret["YQL_TOKEN"],
            },
        )
