# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function

import os
import shutil

from sandbox import sdk2
from sandbox.projects.di.settings import DI_RELEASERS


class AbstractDiResource(sdk2.Resource):
    """Базовый класс для всех ресурсов Интерфейса Дистрибуции."""

    __default_attribute__ = sdk2.parameters.String

    releasable = True
    releasers = DI_RELEASERS
    any_arch = False


class PydiConfigBundle(AbstractDiResource):
    """Архив (или папочка) с конфигами компонент сервиса."""

    any_arch = True
    avoid_suffixes = (".beta",)

    @classmethod
    def is_relevant(cls, fname):
        """Нужно ли копировать конфиг fname? Для прода/тестинга не следует цеплять конфиги чужого окружения."""
        return not any(fname.endswith(suf) for suf in cls.avoid_suffixes)

    @classmethod
    def pack_files(cls, configs_root, task_dir):

        for dir, subdirs, files in os.walk(configs_root):

            def source_path(path):
                """Вычисление полного пути в рамках текущего os.walk()."""
                return os.path.join(dir, path)

            def dest_path(path):
                """
                Вычисление окончательного пути после копирования.

                Мы реализуем руками cp -t task_dir configs_root/*, потому что нужно еще отфильтровать
                конфиги по суффиксу. Поэтому окончательные пути тоже вычисляем сами.
                """
                full_path = source_path(path)
                relpath = os.path.relpath(full_path, configs_root)
                return os.path.join(task_dir, relpath)

            for d in subdirs:
                os.mkdir(dest_path(d))

            for f in files:
                if cls.is_relevant(f):
                    shutil.copyfile(source_path(f), dest_path(f))

        return task_dir


class PydiConfigBundleBeta(PydiConfigBundle):
    """Архив (или папочка) с конфигами компонент сервиса для поднятия на бетах."""
    avoid_suffixes = (".prod",)


class PydiLoopConf(AbstractDiResource):
    """Лупконф для RTC-инсталляции PyDI."""
    file_name = "instancectl/instancectl.conf.prod"


class PydiLoopConfBeta(PydiLoopConf):
    """Лупконф для RTC-инсталляции бет PyDI."""
    file_name = "instancectl/instancectl.conf.beta"


class DiBackendCelery(AbstractDiResource):
    """Бинарник для Celery."""
    executable = True
    arcadia_build_path = "distribution_interface/backend/celery/pydi-celery"


class DiBackendManage(AbstractDiResource):
    """Бинарник для manage.py."""
    executable = True
    arcadia_build_path = "distribution_interface/backend/manage/pydi-manage"


class DiBackendUwsgi(AbstractDiResource):
    """Бинарник для uWSGI."""
    executable = True
    arcadia_build_path = "distribution_interface/backend/uwsgi/pydi-uwsgi"


class DiJinjaRenderer(AbstractDiResource):
    """Процессор шаблонов конфигурационных файлов на этапе препэйринга сервиса."""
    executable = True
    arcadia_build_path = "distribution_interface/rtc_tools/jinja_renderer/jinja_renderer"


class DiUnistat(AbstractDiResource):
    """Юнистат-ручка как микросервис."""
    executable = True
    arcadia_build_path = "distribution_interface/rtc_tools/unistat/bin/unistat"


class DiSandboxSoxTool(AbstractDiResource):
    """
    Штучка, готовящая сырьё для сокс-валидаций.

    В ожидании схождения вот такой вот голсы: https://goals.yandex-team.ru/filter?group=1702&goal=14128
    """
    executable = True
    arcadia_build_path = "distribution_interface/sox_tools/sandbox_tool"


class DiSoxVerifier(AbstractDiResource):
    """Вещица, проверящая код на актуальность и всячески сигнализирующая о неполадках и беспорядках."""
    executable = True
    arcadia_build_path = "distribution_interface/sox_tools/version_verifier/di_version_verifier"


class DiJobs(AbstractDiResource):
    """Запускалка регулярных задач Интерфейса Дистрибуции."""
    executable = True
    arcadia_build_path = "distribution_interface/jobs/bin/di-jobs"


class DiPortoLayer(AbstractDiResource):
    """
    Порто-лэйер для композитного сервиса Интерфейса Дистрибуции.

    Legacy-имя ресурса продиктовано особенностями сборки (где много хардкода и куда не хочется лезть)
    """


class PydiFrontendResource(AbstractDiResource):
    """Собранный архив фронтенда сервиса Интерфейса Дистрибуции."""


class DiBetaFrontendResource(AbstractDiResource):
    """Собранный для бэт архив фронтенда сервиса Интерфейса Дистрибуции."""


class PydiPostgresPackage(AbstractDiResource):
    """
    Архив с полным пакетом Postgres'а (tar.gz).

    Используется в задачах Интерфейса Дистрибуции.
    """
    releasable = False


class PydiMySQLBackup(AbstractDiResource):
    """Полный бэкап production-базы Интерфейса Дистрибуции (tar.gz, mysqldump-формат)."""
    any_arch = True
    releasable = False


class PydiTaxAuditDump(AbstractDiResource):
    """Выгрузка для ФНС."""
    any_arch = True
    releasable = False


class DiAllureReport(AbstractDiResource):
    """Allure отчеты."""
    any_arch = True
    releasable = False
