# -*- coding: utf-8 -*-

import re
import logging
import os
import tempfile

from sandbox import sdk2

from sandbox.projects.sandbox_ci.managers.s3mds import S3Manager
from sandbox.projects.sandbox_ci.utils.request import send_request

URL_REGEXP = re.compile(r"^https://frontend.s3.mds.yandex.net/advertiser-page/\S+/index.html$")


def utf_encode(text):
    return text.encode("utf-8") if type(text).__name__ == 'unicode' else text


class AdvertiserPageRelease(sdk2.Task):
    """Релиз страницы 'О рекламодателе' в продакшн"""

    name = 'ADVERTISER_PAGE_RELEASE'

    s3_bucket_name = 'frontend'

    s3_vault_key_id = 'env.FRONTEND_S3_ACCESS_KEY_ID'

    s3_vault_secret_key = 'env.FRONTEND_S3_SECRET_ACCESS_KEY'

    vault_group = 'FRONTEND'

    key_prefix = 'advertiser-page/production'

    class Requirements(sdk2.Requirements):
        cores = 1  # 1 core
        ram = 1024  # 1GB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        source_url = sdk2.parameters.String(
            'Source url',
            description='Url to index.html',
            required=True,
        )

    def vault_s3_secrets(self):
        s3_key_id = self.s3_vault_key_id
        s3_secret_key = self.s3_vault_secret_key
        vault_group = self.vault_group

        key_id = sdk2.Vault.data(vault_group, s3_key_id)
        access_key = sdk2.Vault.data(vault_group, s3_secret_key)

        return key_id, access_key

    def download_artifact(self):
        logging.info('Starting downloading artefact')

        match = URL_REGEXP.match(self.Parameters.source_url)
        assert match, 'Source url must be in "frontend.s3.mds.yandex.net" bucket and contain "index.html"'

        res = send_request('get', self.Parameters.source_url)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return utf_encode(res.text)

    def save_content_to_file(self, content, tmp_dir):
        logging.info('Starting saving artefact')
        path = os.path.join(tmp_dir, 'index.html')

        with open(path, "w") as f:
            f.write(content)

    def upload_to_mds(self, tmp_dir):
        logging.info('Starting to upload files to mds')
        key_id, access_key = self.vault_s3_secrets()
        client = S3Manager(key_id, access_key, self.s3_bucket_name)
        client.upload_dir(tmp_dir, self.key_prefix)

    def on_execute(self):
        content = self.download_artifact()
        tmp_dir = tempfile.mkdtemp()
        self.save_content_to_file(content, tmp_dir)
        self.upload_to_mds(tmp_dir)
