# -*- coding: utf-8 -*-

import glob
import logging
import os
import os.path

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.config as cc

from sandbox.common.errors import TaskError
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.juggler import resource_types
from sandbox.common.fs import untar_archive

from sandbox.sandboxsdk import environments
from sandbox.projects.common.juggler import jclient

logger = logging.getLogger(__name__)


class LastReleasedAnsibleJugglerResource(sdk2.parameters.Resource):

    resource_type = resource_types.JUGGLER_ANSIBLE.name

    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs={"released": ctt.ReleaseStatus.STABLE},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class AnsiblePlaybook(object):
    def __init__(self, path_to_playbook, path_to_ansible, cwd):
        self.path_to_playbook = path_to_playbook
        self.path_to_ansible = path_to_ansible
        self.cwd = cwd

        self.arguments = ["-v", "-i", "inventory.cfg"]

        self.status = None
        self.stdout = None
        self.stderr = None
        self.summary = {}

        self.logger = logger.getChild(self.__class__.__name__).getChild(os.path.basename(path_to_playbook))

    def run(self):
        for attempt in xrange(1, 4):
            self.logger.info('Attempt %d, run ansible-playbook', attempt)
            process = sp.Popen(
                [self.path_to_ansible, self.path_to_playbook] + self.arguments,
                cwd=self.cwd,
                stdout=sp.PIPE,
                stderr=sp.PIPE,
            )

            stdout, stderr = process.communicate()
            retcode = process.returncode
            self.logger.debug("Stdout: %s" % stdout)
            self.logger.debug("Stderr: %s" % stderr)
            self.logger.debug("Code: %d" % retcode)

            if retcode > 0:
                self.logger.warning('Attempt %d failed', attempt)
            else:
                break

        self.status = retcode
        self.stdout = stdout
        self.stderr = stderr

        if self.stdout:
            self.summary = dict(
                map(lambda x: x.split('='), stdout.strip().split('\n')[-1].split(':')[1].strip().split())
            )

    @property
    def changed(self):
        return int(self.summary.get('changed', 0)) > 0


class DirectRunJugglerPlaybook(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 10800
        # custom parameters
        playbook_repo = sdk2.parameters.String(
            'URL for playbooks', default='https://github.yandex-team.ru/DirectAdmin/ansible-juggler-checks.git'
        )
        source_branch = sdk2.parameters.String('git branch', default='master', required=True)
        playbook_paths = sdk2.parameters.List('Paths to playbook repos', default=['*.yml'])
        token_name = sdk2.parameters.String('Vault secret name with Juggler OAuth token', default='JUGGLER_OAUTH_TOKEN')
        ansible_juggler_resource = LastReleasedAnsibleJugglerResource("Released ansible-juggler")

    class Requirements(sdk2.Task.Requirements):
        environments = sdk2.Requirements.environments.default + (environments.PipEnvironment('httplib2'),)

    class Context(sdk2.Task.Context):
        successfull_playbooks = []

    def preparing(self):
        self.logger = logger.getChild(self.__class__.__name__)

        self.logger.debug("Downloading tar file")
        tar_path = str(sdk2.resource.ResourceData(self.Parameters.ansible_juggler_resource).path)
        venv_path = str(self.path('juggler_ansible'))
        self.logger.debug("Untar resource")
        untar_archive(tar_path, venv_path)

        self.python_path = os.path.join(venv_path, 'bin', 'python')
        self.logger.debug("Python path: %s", self.python_path)
        self.ansible_playbook_path = os.path.join(venv_path, 'bin', 'ansible-playbook')
        self.logger.debug("ansible-playbook path: %s", self.ansible_playbook_path)

        self.checkout_path = str(self.path(os.path.basename(self.Parameters.playbook_repo)))
        self.logger.debug("Git clone to %s", self.checkout_path)
        git = Git(self.Parameters.playbook_repo)
        git.clone(self.checkout_path, self.Parameters.source_branch)

        self.logger.debug("Create empty hosts file")
        inventory_path = os.path.join(self.checkout_path, 'hosts')
        with open(inventory_path, 'w'):
            pass

        juggler_token = sdk2.Vault.data(self.owner, self.Parameters.token_name)
        os.environ['JUGGLER_OAUTH_TOKEN'] = juggler_token

    def on_execute(self):
        self.preparing()
        self.apply_playbooks()

    def apply_playbooks(self):
        # Find file into checkout_path but return path to file without checkout_path
        playbooks = sorted(
            [
                f.replace(self.checkout_path + '/', '')
                for g in self.Parameters.playbook_paths
                for f in glob.iglob(os.path.join(self.checkout_path, g))
            ]
        )
        self.logger.debug("Total playbooks after globbing (%d): %s", len(playbooks), str(playbooks))

        playbooks_to_process = sorted(list(set(playbooks) - set(self.Context.successfull_playbooks)))
        self.logger.debug("Playbooks to process (%d): %s", len(playbooks_to_process), str(playbooks_to_process))

        playbooks = []
        failed_playbooks = []
        changed_playbooks = []
        for filename in playbooks_to_process:
            playbook = AnsiblePlaybook(filename, self.ansible_playbook_path, self.checkout_path)
            playbooks.append(playbook)
            playbook.run()

            if playbook.status == 0:
                self.Context.successfull_playbooks.append(filename)
            else:
                failed_playbooks.append(playbook)

            if playbook.changed:
                changed_playbooks.append(playbook)

        msg = ["Applied %d playbooks" % len(playbooks)]

        if changed_playbooks:
            msg.append("Got a changes in playbooks:")
            for playbook in changed_playbooks:
                msg.append("\t- %s: %s" % (playbook.path_to_playbook, playbook.summary))

        if failed_playbooks:
            msg.append("Failed playbooks:")
            for playbook in failed_playbooks:
                msg.append("\t- %s: %s" % (playbook.path_to_playbook, playbook.summary))

        self.set_info("\n".join(msg))

        if failed_playbooks:
            raise TaskError('Failed to apply playbooks: ' + ', '.join([p.path_to_playbook for p in failed_playbooks]))

        if cc.Registry().common.installation != ctm.Installation.LOCAL:
            jclient.send_events_to_juggler('direct-sandbox', "ansible-juggler", 'OK', 'Passed')
        else:
            self.logger.debug("Don't send monitoring because of local installation")
