# -*- coding: utf-8 -*-

import logging
import os

import requests

import sandbox.common.types.misc as ctm
import sandbox.projects.common.environments as environments
import sandbox.sdk2 as sdk2
import sandbox.sdk2.helpers as helpers
import sandbox.sdk2.vcs.git as git
import sandbox.sdk2.vcs.svn as svn

logger = logging.getLogger(__name__)


class BuildMavenProject(sdk2.Task):
    """Таск для сборки мавена из исходников"""

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = (environments.SandboxJavaJdkEnvironment('1.8.0'), environments.SandboxMavenEnvironment("3.2.2"))

    class Parameters(sdk2.Parameters):
        kill_timeout = 3000
        ssh_key = sdk2.parameters.Vault('SSH key in Vault ("name" or "owner:name")', required=True)
        with sdk2.parameters.Group("VCS settings") as vcs_settings:
            repo = sdk2.parameters.String("Repo URL or path in Arcadia")
            branch = sdk2.parameters.String("Repo branch (for git)", default="master", required=False)
        with sdk2.parameters.Group("Maven settings") as maven_settings:
            targets = sdk2.parameters.Dict(
                "Key: path to pom.xml; Value: semicolon separated maven targets", default={"pom.xml": ""}, required=True
            )
            settings = sdk2.parameters.Url(
                "Settings file URL",
                default='https://github.yandex-team.ru/raw/qatools/settings.xml/master/settings.xml',
                required=False,
            )

    def on_prepare(self):
        self.logger = logger.getChild(self.__class__.__name__)
        self.source_path = str(self.path())
        self._get_source()
        self.settings_path = self._download_settings()
        self.maven_log_num = 0

    def on_execute(self):
        for pom_file, targets in self.Parameters.targets.items():
            for target in targets.split(';'):
                self._run_maven(pom_file, target.strip())

    def _get_source(self):
        """Общий метод для выкачивания исходников в зависимости от VCS"""
        schema = self.Parameters.repo[0:3]
        if schema == 'git':
            self._checkout_git()
        elif schema == 'svn':
            self._checkout_svn()
        else:
            raise ValueError("Unknown schema {}, only support svn over ssh and git over ssh".format(schema))

    def _checkout_svn(self):
        """Выкачивает исходники из svn"""
        with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
            svn.Svn.checkout(self.Parameters.repo, self.source_path)

    def _checkout_git(self):
        """Выкачивает исходники из git"""
        with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
            git.Git(self.Parameters.repo).clone(self.source_path, self.Parameters.branch)

    def _download_settings(self):
        """Скачивает файл настроек для мавена с указанного URL, кидает RuntimeError если не может скачать"""
        if not self.Parameters.settings:
            return None
        settings_path = os.path.join(str(self.source_path), 'settings.xml')
        r = requests.get(self.Parameters.settings)
        r.raise_for_status()
        with open(settings_path, 'w') as fd:
            fd.write(r.content)
        return settings_path

    def _run_maven(self, pom_file, target):
        self.maven_log_num += 1
        cmd = ['mvn', '-l', "{}/maven_{}.log".format(self.agentr.logdir, self.maven_log_num)]
        cmd += ['-s', self.settings_path] if self.settings_path else []
        cmd += ['-Djava.net.preferIPv6Addresses=true']
        cmd += ['-f', pom_file]
        cmd += target.split()
        cmd = [str(x) for x in cmd]
        self.logger.debug("Running command: {}".format(cmd))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("maven")) as pl:
            sp = helpers.subprocess
            with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
                retcode = sp.Popen(cmd, shell=False, stdout=pl.stdout, stderr=sp.STDOUT).wait()
                if retcode:
                    raise RuntimeError("maven failed")
