# -*- coding: utf-8 -*-
import logging
import os
import re
import platform
import stat
import time
import shutil

from sandbox import common
from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox.projects.porto import common as porto_common
from sandbox.projects.resource_types import PLAIN_TEXT as PlainText

from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia, Svn

from sandbox.common.errors import TaskFailure
from sandbox.projects.common.vcs import arc
from sandbox.projects.common import error_handlers as eh

from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, image


class UseArcFuse(sdk2.parameters.Bool):
    name = "use_arc_fuse"
    description = "Choose to use arc fuse api"
    default = True


class ArcRevision(sdk2.parameters.String):
    name = "arc_revision"
    description = "Ex: r7312366, users/dim-gonch/logos, trunk, fd6aa52239c"
    default_value = "trunk"


class ArcToken(sdk2.parameters.YavSecret):
    name = "arc_token"
    description = "Arc OAuth token, if empty get from vault ARC_TOKEN"
    default_value = ""


class SvnRevision(sdk2.parameters.String):
    name = "svn_revision"
    description = "Arcadia trunk revision"
    default_value = "trunk"


class UbuntuRelease(sdk2.parameters.String):
    name = "ubuntu_release"
    description = "Ubuntu release"
    default_value = image.UbuntuRelease.BIONIC

    @common.utils.singleton_classproperty
    def choices(self):
        resource_types = []
        for rt in image.UbuntuRelease:
            resource_types.append((rt, rt))
        return resource_types

    required = True


class BuildLxc(sdk2.parameters.Bool):
    name = "build_lxc"
    description = "Build LXC-container instead of Porto layer"
    default = False


class ParentLayerType(sdk2.parameters.String):
    name = "parent_layer_type"
    description = "Parent layer type"
    default_value = ""

    @common.utils.singleton_classproperty
    def choices(self):
        return [("None", "")] + [(rt, rt) for rt in porto_common.porto_layer_types()]

    required = False


class ParentLayer(sdk2.parameters.Resource):
    name = "parent_layer"
    description = "Parent layer (default: latest stable release)"

    @common.utils.singleton_classproperty
    def resource_type(self):
        return porto_common.porto_layer_types()

    required = False


class PortoLayerType(sdk2.parameters.String):
    name = "layer_type"
    description = "Layer resource type"
    default_value = "PORTO_LAYER"

    @common.utils.singleton_classproperty
    def choices(self):
        return [(rt, rt) for rt in porto_common.porto_layer_types() + porto_common.porto_image_types()]

    required = True


class LxcLayerType(sdk2.parameters.String):
    name = "layer_type"
    description = "Container resource type"
    default_value = "LXC_CONTAINER"

    @common.utils.singleton_classproperty
    def choices(self):
        resource_types = []
        for rt in sdk2.Resource:
            if rt.name.startswith("LXC_CONTAINER") or rt.name.endswith("LXC_CONTAINER"):
                resource_types.append((rt.name, rt.name))
        return resource_types

    required = True


class LayerName(sdk2.parameters.String):
    name = "layer_name"
    description = "Layer name"
    default_value = ""
    required = True


class Compress(sdk2.parameters.String):
    name = "compress"
    description = "Compression"
    default_value = "tar.xz"
    choices = [
        ("tar", "tar"),
        ("tar.gz", "tar.gz"),
        ("tar.xz", "tar.xz"),
        ("tar.zst", "tar.zst"),
    ]
    required = True


class ScriptUrl(sdk2.parameters.Url):
    name = "script"
    description = "Setup scripts URLs (arcadia:/path[@rev] or text without auth via https://)"
    default_value = ""
    required = False
    multiline = True


class LxcScriptUrl(ScriptUrl):
    pass


class PortoScriptUrl(ScriptUrl):
    pass


class Script2Url(sdk2.parameters.Url):
    name = "script2"
    description = "Post OS setup scripts URLs"
    default_value = ""
    required = False
    multiline = True


class ScriptEnv(sdk2.parameters.Dict, sdk2.parameters.String):
    name = "script_env"
    description = "Environment variables"
    default_value = None


class StartOS(sdk2.parameters.Bool):
    name = "start_os"
    description = "Start OS for scripts"
    default_value = True
    required = True


class KeepResolvConf(sdk2.parameters.Bool):
    name = "keep_resolv_conf"
    description = "Manual /etc/resolv.conf"
    default_value = False
    required = False


class MergeLayers(sdk2.parameters.Bool):
    name = "merge_layers"
    description = "Merge all layers"
    default_value = False
    required = False


class SpaceLimit(sdk2.parameters.Integer):
    name = "space_limit"
    description = "Max disk usage (MB)"
    default_value = 8192
    required = False


class MemoryLimit(sdk2.parameters.Integer):
    name = "memory_limit"
    description = "Max memory usage (MB)"
    default_value = 4096
    required = False


class DebugBuild(sdk2.parameters.Bool):
    name = "debug_build"
    description = "Keep volume and container for manual debugging"
    default_value = False
    required = False


class OutputResourceID(sdk2.parameters.Integer):
    name = "output_resource_id"
    description = "ID of resource to be filled with output layer"
    required = False
    default_value = None


class OutputResourceAttr(sdk2.parameters.Dict, sdk2.parameters.String):
    name = "output_resource_attr"
    description = "Output resource attributes"
    default_value = None


class BuildPortoLayer2(sdk2.Task):
    """
    Build proto-layer or LXC-container with optional parent layer via
    bash scripts from Arcadia or from HTTPS with arc and yav support

    :rtype: `sdk2.Task`
    """

    class Requirements(sdk2.Requirements):
        privileged = True  # Require root permissions (container in privileged mode)
        client_tags = ctc.Tag.PORTOD
        dns = ctm.DnsType.DNS64  # Enable NAT64 to access to IPv4-only hosts

    class Parameters(sdk2.Task.Parameters):
        use_arc_fuse = UseArcFuse("Use Arc VCS")

        with sdk2.parameters.Group("VCS params") as arc_params_group:
            with use_arc_fuse.value[True]:
                arc_revision = ArcRevision("Arcadia branch / commit hash / revision")
                arc_token = ArcToken("Arc token")
            with use_arc_fuse.value[False]:
                svn_revision = SvnRevision("Svn revision")

        with sdk2.parameters.Group("Porto params") as lxc_params_gropup:
            build_lxc = BuildLxc()

            with build_lxc.value[True]:
                ubuntu_release = UbuntuRelease()
                lxc_layer_type = LxcLayerType()
                lxc_script = LxcScriptUrl()

            with build_lxc.value[False]:
                parent_layer_type = ParentLayerType()
                parent_layer = ParentLayer()
                porto_layer_type = PortoLayerType()
                layer_name = LayerName()
                porto_script = PortoScriptUrl()
                script2 = Script2Url()
                script_env = ScriptEnv("Environment variables")
                compress = Compress()
                start_os = StartOS()
                keep_resolv_conf = KeepResolvConf()
                merge_layers = MergeLayers()
                space_limit = SpaceLimit()
                memory_limit = MemoryLimit()
                debug_build = DebugBuild()

        output_resource_attr = OutputResourceAttr("Output resource attributes")

    class Context(sdk2.Task.Context):
        children = []

    def remote_copy(
        self,
        src,
        dst,
        protocol=None,
        unstable=None,
        exclude=None,
        timeout=None,
        create_resource=False,
        resource_descr=None,
        write_to_task_logs=True,
    ):
        if unstable is None:
            unstable = []
        if exclude is None:
            exclude = []
        if timeout is None:
            timeout = 3 * 60 * 60
        if protocol is None:
            protocol = "copy"
            if ":" in src:
                temp = src.split(":")[0]
                if temp == "svn+ssh":
                    protocol = "svn"
                elif temp == "arcadia":
                    protocol = "arcadia"
                elif temp in ["http", "https"]:
                    protocol = "http"
                elif temp in ["rbtorrent", "sbr"]:
                    protocol = "skynet"

        #############
        if protocol == "copy":  # Just copy files from SVN or Arc mount point
            logging.info("Copy file from repo: {} to: {}".format(src, dst))
            shutil.copyfile(os.path.join(self._mount_dir, src), str(dst))
        elif protocol in ("rcp", "scp"):
            with sdk2.helpers.ProcessLog(self, logger="remote_copy" if write_to_task_logs else None) as pl:
                sp.Popen([protocol, "-r", src, dst], stdout=pl.stdout, stderr=pl.stderr).wait(timeout)

            # Convert newlines in scripts to universal newline
            dst_text = open(str(dst), "U").read()
            open(str(dst), "w").write(dst_text)

        elif protocol == "http":
            cmd = {
                "linux": "wget -T 60 -O {} {}",
                "darwin": "curl -m 60 -o {} {}",
                "freebsd": "fetch -T 60 -o {} {}",
            }[platform.uname()[0].lower()]

            with sdk2.helpers.ProcessLog(self, logger="remote_copy" if write_to_task_logs else None) as pl:
                sp.Popen(cmd.format(dst, src), stdout=pl.stdout, stderr=pl.stderr, shell=True).wait(timeout)

            # Convert newlines in scripts to universal newline
            dst_text = open(str(dst), "U").read()
            open(str(dst), "w").write(dst_text)

        elif protocol == "svn":
            Svn.export(src, dst)

        elif protocol == "arcadia":
            Arcadia.export(src, dst)

        elif protocol == "skynet":
            # check if we get torrent id
            if src.startswith("rbtorrent:"):
                common.share.skynet_get(src, dst, timeout)
            else:
                common.share.skynet_copy(
                    src.partition(":")[0], src.partition(":")[2], dst, unstable=unstable, exclude=exclude
                )
        else:
            raise common.errors.TaskError("unknown remote file protocol {}".format(protocol))

        ############
        if create_resource:
            resource = PlainText(self, resource_descr, dst)
            data = sdk2.ResourceData(resource)
            data.ready()
            return resource

    def _parent_resources(self):
        parent_resources = []
        resource_id = self.Parameters.parent_layer
        parent_layer_type = self.Parameters.parent_layer_type

        if not resource_id and parent_layer_type:
            resource_id = (
                sdk2.Resource.find(type=parent_layer_type, attrs={"arch": "linux", "released": "stable"}).first().id
            )
            self.set_info("Use latest stable release of {} id {}".format(parent_layer_type, resource_id))

        while resource_id:
            resource = sdk2.Resource[resource_id]
            self.set_info("Parent layer {} id {}".format(resource.type, resource.id))
            parent_resources.append(resource)

            if not hasattr(resource, "parent_layer"):
                break

            resource_id = int(getattr(resource, "parent_layer"))
        return parent_resources

    @staticmethod
    def _checkout_svn(revision):
        logging.debug("Checkout SVN rev: {}".format(revision))
        if revision == "trunk":
            revision = ""

        for i in range(10):
            try:
                svn_dir = Arcadia.checkout(
                    Arcadia.trunk_url(revision=revision),
                    "svn_dir",
                )
                break
            except Exception:
                logging.warning("Checkout failed SVN rev: {}. Try: [{}/10]".format(revision, i + 1))
                if i == 9:
                    raise
                time.sleep(5)
        return svn_dir

    @staticmethod
    def _mount_arc(arc_token, mount_point, changeset, fetch_all=False):
        arc_cli = arc.Arc(arc_oauth_token=arc_token)
        arcadia_src_dir = os.path.abspath(mount_point)
        for i in range(10):
            try:
                mp = arc_cli.mount_path(None, mount_point=mount_point, changeset=changeset, fetch_all=fetch_all)
                if mp.mounted:
                    logging.debug("Arc Mounted: {}".format(arcadia_src_dir))
                    break
                raise ValueError("not mounted")
            except Exception as e:
                logging.warning("Arc mount failed. Error: {} | Try: [{}/10]".format(e, i + 1))
                if i == 9:
                    raise TaskFailure("Can't mount arcadia: {}, rev={}".format(arcadia_src_dir, changeset))
                time.sleep(5)
        return arcadia_src_dir, mp

    def _mount_repo(self):
        mount_dir = None
        mp = None
        if not self.Parameters.use_arc_fuse:  # use SVN
            mount_dir = self._checkout_svn(revision=self.Parameters.svn_revision)
        else:  # use Arc
            arc_token = None
            if self.Parameters.arc_token:
                default_key = self.Parameters.arc_token.default_key or "arc_token"
                yav_data = self.Parameters.arc_token.data()
                eh.ensure(default_key in yav_data, "No such key `{}` found in `arc_token`".format(default_key))
                arc_token = yav_data[default_key]
            mount_dir, mp = self._mount_arc(
                arc_token=arc_token, mount_point="arc_dir", changeset=self.Parameters.arc_revision
            )
        return mount_dir, mp

    def build_lxc(self):
        ubuntu_release = self.Parameters.ubuntu_release
        layer_type = self.Parameters.lxc_layer_type

        scripts = []

        for script_url in filter(None, re.split("[\r\n]+", self.Parameters.lxc_script)):
            script_path = self.path("script{}.sh".format(len(scripts)))
            self.remote_copy(script_url, script_path, create_resource=True, resource_descr=script_url)
            scripts += [(script_url, script_path)]

        custom_script = []
        for script_url, script_path in scripts:
            script = []
            script.append("# -------------------- start {} -------------------- #".format(script_url))
            script.append(open(str(script_path)).read())
            script.append("# -------------------- end {} -------------------- #".format(script_url))
            custom_script.append("\n".join(script))

        sli_subtask = SandboxLxcImage(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,  # Owner is required
            resource_type=layer_type,
            ubuntu_release=ubuntu_release,
            test_result_lxc=True,
            custom_image=True,
            custom_attrs=self.Parameters.output_resource_attr,
            custom_script="\n\n".join(custom_script),
        )

        self.Context.children.append(sli_subtask.save().enqueue().id)  # Save subtask ID and enqueue it

        raise sdk2.WaitTask(self.Context.children, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

    def fetch_lxc_resource(self):
        ubuntu_release = self.Parameters.ubuntu_release
        layer_type = self.Parameters.lxc_layer_type

        for task_id in self.Context.children:
            lxc_container = sdk2.Resource.find(task_id=task_id, type=layer_type).first()
            data = sdk2.ResourceData(lxc_container)

            resource_attributes = {}
            if self.Parameters.output_resource_attr:
                resource_attributes.update(self.Parameters.output_resource_attr)

            lxc_container_name = os.path.basename(str(data.path))
            dst = self.path(lxc_container_name)
            shutil.copyfile(str(data.path), str(dst))

            packed_size = os.path.getsize(str(dst))
            logging.info("Packed size {}M".format(packed_size >> 20))
            self.set_info("Packed size {}M".format(packed_size >> 20))

            resource_attributes["name"] = lxc_container_name
            resource_attributes["description"] = lxc_container.description
            resource_attributes["arch"] = "linux"
            resource_attributes["ubuntu_release"] = ubuntu_release
            resource_attributes["type"] = "IMAGE"

            logging.info("Create resource type: {} path: {}".format(layer_type, dst))
            resource = sdk2.Resource[layer_type](task=self, path=dst, **resource_attributes)
            data = sdk2.ResourceData(resource)
            data.ready()  # Set resource status to READY

    def build_porto(self):
        parent_layer = self.Parameters.parent_layer
        parent_layer_type = self.Parameters.parent_layer_type
        layer_type = self.Parameters.porto_layer_type
        layer_name = self.Parameters.layer_name

        compress = self.Parameters.compress
        output_file = "{}.{}".format(layer_name, compress)
        build_image = layer_type.startswith("PORTO_IMAGE")
        build_qcow = layer_type.startswith("QEMU_IMAGE")
        build_squash = layer_type.startswith("PORTO_SQUASH")
        start_os = self.Parameters.start_os
        merge_layers = self.Parameters.merge_layers
        debug_build = self.Parameters.debug_build

        scripts = []
        env = {}

        cmd = ["portoctl", "--timeout", "0", "build"]

        for script_url in filter(None, re.split("[\r\n]+", self.Parameters.porto_script)):
            script_path = self.path("script{}.sh".format(len(scripts)))
            self.remote_copy(script_url, script_path, create_resource=True, resource_descr=script_url)
            scripts += [(script_url, script_path)]

            cmd += ["-S", str(script_path)]

        for script_url in filter(None, re.split("[\r\n]+", self.Parameters.script2)):
            script_path = self.path("script{}.sh".format(len(scripts)))
            self.remote_copy(script_url, script_path, create_resource=True, resource_descr=script_url)
            scripts += [(script_url, script_path)]

            cmd += ["-s", str(script_path)]

        parent_resources = self._parent_resources()
        for resource in parent_resources:
            layer_path_tmp = sdk2.ResourceData(resource).path
            cmd += ["-L", str(layer_path_tmp)]

        if build_squash:
            output_file = layer_name + ".squash"
            cmd += ["-Q", output_file]
        elif build_image:
            cmd += ["-O", layer_name + ".img"]
        elif build_qcow:
            output_file = layer_name + ".img"
            cmd += ["-O", layer_name + ".raw"]
        else:
            cmd += ["-o", output_file]

        if merge_layers:
            cmd += ["-M"]

        # drop cap_sys_admin in OS container
        cmd += ["-D"]

        # keep volume and container for manual debugging
        if debug_build:
            cmd += ["-k"]

        if self.Parameters.script_env:
            env.update(self.Parameters.script_env)

        env["sandbox_task_id"] = self.id

        if env:
            cmd += ["env=" + ";".join(["{}={}".format(k, v).replace(";", r"\;") for k, v in env.iteritems()])]

        cmd += ["bind_dns=false"]

        if self.Parameters.keep_resolv_conf:
            cmd += ["resolv_conf=keep"]

        # TODO: net
        cmd += ["net=inherited"]

        cmd += [
            "aging_time=3600",
            "enable_porto=false",
            "hostname=sandbox-{}".format(self.id),
            "virt_mode=" + ("os" if start_os else "app"),
            "private=sandbox-{}".format(self.id),
            "space_limit={}M".format(self.Parameters.space_limit),
            "memory_limit={}M".format(self.Parameters.memory_limit),
        ]

        place_dir = self.path("porto_place")
        place_dir.mkdir()
        for subdir in ("porto_layers", "porto_volumes", "porto_storage"):
            os.mkdir(os.path.join(str(place_dir), subdir))
        cmd += ["place=" + str(place_dir)]

        with sdk2.helpers.ProcessLog(logger="build") as pl:
            try:
                sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)
            except sp.CalledProcessError as e:
                try:
                    self.set_info("".join(open(self.path("build.out.log")).readlines()[-50:]))
                except:
                    pass
                raise e

        if build_image:
            with sdk2.helpers.ProcessLog(self, logger="tar-img") as pl:
                sp.check_call(["tar", "cSaf", output_file, layer_name + ".img"], stdout=pl.stdout, stderr=pl.stderr)

        if build_qcow:
            for i in range(256):
                dev_name = "/dev/loop{}".format(i)
                if not os.path.exists(dev_name):
                    os.mknod(dev_name, stat.S_IFBLK | 0o644, os.makedev(7, i))
            volume_path = self.path("image")
            disk_img = self.path("disk.raw")
            image_size = os.path.getsize(layer_name + ".raw")
            gptbin = ""
            gptbin_search = [
                "/usr/lib/syslinux/gptmbr.bin",
                "/usr/lib/EXTLINUX/gptmbr.bin",
                "/usr/lib/SYSLINUX/gptmbr.bin",
            ]
            for f in gptbin_search:
                if os.path.exists(f):
                    gptbin = f
            if not gptbin:
                raise Exception("Can not fild bootloader stage1 binnary: gptmbr.bin")

            di = open(str(disk_img), "a")
            di.truncate(image_size + (2 << 20))
            di.close()
            # Create GPT partitions
            # NOTE: Do not set partition label name because of https://st.yandex-team.ru/QEMUKVM-379
            with sdk2.helpers.ProcessLog(self, logger="qemu-image") as pl:
                sp.check_call(
                    ["sgdisk", "--clear", "-g", "--new", "1::-0", "--typecode=1:8300", str(disk_img)],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                sp.check_call(["sgdisk", "--attributes=1:set:2", str(disk_img)], stdout=pl.stdout, stderr=pl.stderr)

                # Install bootloader stage1
                sp.check_call(
                    ["dd", "if=" + gptbin, "of=" + str(disk_img), "conv=notrunc", "bs=440", "count=1"],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                sp.check_call(
                    [
                        "dd",
                        "if=" + layer_name + ".raw",
                        "of=" + str(disk_img),
                        "conv=notrunc",
                        "bs=1048576",
                        "seek=1",
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                p = sp.Popen(["losetup", "--find", "--show", str(disk_img)], stdout=sp.PIPE, stderr=pl.stderr)
                loopdev = p.stdout.readline().strip()
                sp.check_call(["partprobe", loopdev], stdout=pl.stdout, stderr=pl.stderr)
                # Double check that partprobe have found partition
                if not os.path.exists(loopdev + "p1"):
                    os.mknod(loopdev + "p1", stat.S_IFBLK | 0o644, os.makedev(259, int(loopdev[len("/dev/loop") :])))
                # Create ext4 journal
                sp.check_call(["tune2fs", "-O", "has_journal", loopdev + "p1"], stdout=pl.stdout, stderr=pl.stderr)
                # Install bootloader stage2
                volume_path.mkdir()
                sp.check_call(["mount", loopdev + "p1", str(volume_path)], stdout=pl.stdout, stderr=pl.stderr)
                sp.check_call(["extlinux", "--install", str(volume_path)], stdout=pl.stdout, stderr=pl.stderr)
                sp.check_call(["umount", loopdev + "p1"], stdout=pl.stdout, stderr=pl.stderr)
                volume_path.rmdir()
                sp.check_call(["tune2fs", "-L", "rootfs", loopdev + "p1"], stdout=pl.stdout, stderr=pl.stderr)
                sp.check_call(
                    ["e2fsck", "-f", "-p", "-E", "discard", loopdev + "p1"], stdout=pl.stdout, stderr=pl.stderr
                )
                sp.check_call(["losetup", "-d", loopdev], stdout=pl.stdout, stderr=pl.stderr)
                sp.check_call(
                    ["qemu-img", "convert"]
                    + (["-c"] if Compress.name != "tar" else [])
                    + ["-f", "raw", "-O", "qcow2", str(disk_img), output_file],
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                os.unlink(layer_name + ".raw")
                disk_img.unlink()

        resource_attributes = {}
        if self.Parameters.output_resource_attr:
            resource_attributes.update(self.Parameters.output_resource_attr)

        if not merge_layers and not build_image and parent_layer:
            resource_attributes["parent_layer"] = int(parent_layer)
            resource_attributes["dependencies"] = " ".join([str(r.id) for r in parent_resources])

        resource_attributes["name"] = layer_name
        resource_attributes["stack"] = layer_name
        resource_attributes["compression"] = compress
        for resource in parent_resources:
            if hasattr(resource, "name"):
                resource_attributes["stack"] += " " + getattr(resource, "name", "unknown")

        packed_size = os.path.getsize(output_file)
        try:
            unpacked_size = int(
                sp.check_output(["getfattr", "-n", "user.porto.unpacked_size", "--only-values", output_file])
            )
        except:
            unpacked_size = 0

        logging.info("Packed size {}M, unpacked size {}M".format(packed_size >> 20, unpacked_size >> 20))
        self.set_info("Packed size {}M, unpacked size {}M".format(packed_size >> 20, unpacked_size >> 20))

        if unpacked_size:
            resource_attributes["unpacked_size"] = str(unpacked_size)

        resource_attributes["description"] = "Porto-layer with type {} based on {} layer".format(
            layer_type, parent_layer_type
        )

        logging.info("Create resource type: {} path: {}".format(layer_type, output_file))
        resource = sdk2.Resource[layer_type](task=self, path=output_file, **resource_attributes)
        data = sdk2.ResourceData(resource)
        data.ready()  # Set resource status to READY

    def on_execute(self):
        script_dir = str(self.path("script").absolute())
        logging.info("Script dir: {}".format(script_dir))
        os.makedirs(script_dir)

        self._mount_dir, arc_mount = self._mount_repo()
        logging.debug("Repo Mounted: {}".format(self._mount_dir))

        if self.Parameters.build_lxc:  # Build LXC-container
            with self.memoize_stage.build_lxc(commit_on_entrance=False):
                self.build_lxc()

            self.fetch_lxc_resource()
        else:  # Build Porto-layer
            self.build_porto()

        if arc_mount and arc_mount.mounted:
            arc_mount.unmount()
