# -*- coding: utf-8 -*-

import logging

import requests
import yaml

import create_subtasks
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.projects.common.environments as environments
import sandbox.sdk2 as sdk2
import sandbox.sdk2.vcs.svn as svn
from resolve_dependencies import resolve_dependencies

logger = logging.getLogger(__name__)

ARCADIA_BASE_PATH = "svn+ssh://arcadia.yandex.ru/arc/"


class DirectBuildAutotests(sdk2.Task):
    """Таск для сборки автотестов директа"""

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = (environments.SandboxJavaJdkEnvironment('1.8.0'), environments.SandboxMavenEnvironment("3.2.2"))

    class Parameters(sdk2.Parameters):
        kill_timeout = 3000

        projects_names = sdk2.parameters.List("Projects names", required=True)

        with sdk2.parameters.Group("Repository settings") as repo_settings:
            with sdk2.parameters.RadioGroup("Repository") as repo_type:
                repo_type.values.github = repo_type.Value("GitHub: https://github.yandex-team.ru", default=True)
                repo_type.values.arcadia = "Arcadia: https://arcadia.yandex.ru/arc"
                with repo_type.value.github:
                    github_organization = sdk2.parameters.String(
                        "Github organization", default="direct-qa", required=True
                    )
                    branch = sdk2.parameters.String("Git branch", default="master", required=True)
                    build_config = sdk2.parameters.Url(
                        "YAML build config URL",
                        default="https://raw.github.yandex-team.ru/direct-qa/jenkins-dsl/master/directci/config.yml",
                        required=True,
                    )
                with repo_type.value.arcadia:
                    arcadia_path = sdk2.parameters.String(
                        "Arcadia relative path", default="trunk/arcadia/junk/ya-bogdan/direct-qa", required=True
                    )
                    arcadia_build_config = sdk2.parameters.String(
                        "YAML build config Arcadia relative path",
                        default="trunk/arcadia/junk/ya-bogdan/direct-qa/config.yml",
                        required=True,
                    )
                # TODO: add it to arcadia sometime in the future...
                settings = sdk2.parameters.String(
                    "Maven settings URL",
                    default="https://raw.github.yandex-team.ru/qatools/settings.xml/master/settings.xml",
                    required=True,
                )

        ssh_key = sdk2.parameters.Vault(
            'SSH key in Vault ("name" or "owner:name")',
            required=True,
            default="robot-irt-jenkins:robot-irt-jenkins-ssh",
        )

        with sdk2.parameters.Group('Internal settings') as internal_settings:
            rebuild_downstream = sdk2.parameters.Bool("Rebuild downstream", required=True, default=True)
            with sdk2.parameters.RadioGroup("Maven target") as maven_target:
                maven_target.values.deploy = maven_target.Value('deploy', default=True)
                maven_target.values.compile = 'compile'
            with sdk2.parameters.RadioGroup("DB stage (for DirectDbModels only)") as db_stage:
                db_stage.values.ts = 'ts'
                db_stage.values.ts2 = 'ts2'
                db_stage.values.devtest = 'devtest'
                db_stage.values.dev7 = 'dev7'

    def on_prepare(self):
        self.logger = logger.getChild(self.__class__.__name__)

        # Обработка аркадийных параметров
        self.arcadia_path = ARCADIA_BASE_PATH + self.Parameters.arcadia_path
        self.build_config = (
            ARCADIA_BASE_PATH + self.Parameters.arcadia_build_config
            if self.Parameters.repo_type == "arcadia"
            else self.Parameters.build_config
        )

        # Получение YAML-конфига для паков
        self.config = self._get_yaml_config()

        # Получение URL-ов для паков
        self.source_urls = self._get_all_repos_urls_by_name()

    def _get_yaml_config(self):
        if self.Parameters.repo_type == "github":
            r = requests.get(self.build_config)
            r.raise_for_status()
            config = r.content
        elif self.Parameters.repo_type == "arcadia":
            with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
                config = svn.Svn.cat(self.build_config)
        return yaml.load(config)

    def _get_all_repos_urls_by_name(self):
        if self.Parameters.repo_type == "github":
            all_repos = [
                requests.get(
                    'https://api.github.yandex-team.ru/orgs/{}/repos?page={}'.format(
                        self.Parameters.github_organization, page
                    )
                )
                for page in range(1, 20)
            ]
            return {repo['name']: repo['ssh_url'] for page in all_repos for repo in page.json()}
        elif self.Parameters.repo_type == "arcadia":
            with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
                all_repos = svn.Svn.list(self.arcadia_path, as_list=True)
            path = self.arcadia_path + '/' if self.arcadia_path[-1] != '/' else self.arcadia_path
            return {repo[:-1]: path + repo[:-1] for repo in all_repos if repo[-1] == '/'}  # выбираем только директории

    def on_execute(self):
        with self.memoize_stage.dependencies:
            self.Context.levels = resolve_dependencies(
                self.config,
                self.Parameters.projects_names,
                self.Parameters.maven_target,
                self.Parameters.rebuild_downstream,
            )
            self.logger.info("Build queue: {}".format(self.Context.levels))
            self.Context.current_level = 0
            for packs in self.Context.levels:
                for pack, goal in packs:
                    if pack not in self.source_urls:
                        raise RuntimeError("Pack {} not found at repository".format(pack))

        if self.Context.current_level > 0:
            self._check_children_tasks()
        if self.Context.current_level < len(self.Context.levels):
            self._build_packs()

    def _build_packs(self):
        self.Context.task_ids = []
        self.Context.tasks_to_rebuild = []
        packs = self.Context.levels[self.Context.current_level]
        self.logger.info("Build {} level of dependencies, packs are: {}".format(self.Context.current_level, packs))
        for pack, goal in packs:
            builder_class = self.config[pack].get('builderClass')
            if builder_class == 'StepsAndTests':
                self._build_steps_and_tests(pack, goal)
            elif builder_class == 'SwaggerApiClient':
                self._build_swagger_api_client(pack, goal)
            elif builder_class == 'DirectDbModels':
                self._build_direct_db_models(pack, goal)
            else:
                self._build_regular(pack, goal)
        self.Context.current_level += 1
        raise sdk2.WaitTask(self.Context.task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _check_children_tasks(self):
        """Проверяет статусы завершившихся подзадач.

        Если есть неуспешные, то пробует их перезапустить. Если после перезапуска подзадача
        так и не выполнилась успешно, то завершает родительскую задачу с ошибкой.
        """
        if self.Context.tasks_to_rebuild:
            for task_id in self.Context.tasks_to_rebuild:
                task_result = sdk2.Task[task_id]
                if task_result.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure("One or more subtasks failed after retrying")
            self.Context.tasks_to_rebuild = []
        for task_id in self.Context.task_ids:
            task_result = sdk2.Task[task_id]
            if task_result.status != ctt.Status.SUCCESS:
                self.Context.tasks_to_rebuild.append(task_id)
                self.logger.warn("Subtask #{} failed, trying to rerun".format(task_id))
                task_result.enqueue()
        if self.Context.tasks_to_rebuild:
            self.Context.task_ids = []
            raise sdk2.WaitTask(
                self.Context.tasks_to_rebuild, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True
            )

    def _build_steps_and_tests(self, pack, goal):
        task = create_subtasks.create_maven_build_subtask(
            self,
            self.source_urls[pack],
            {
                'steps/pom.xml': 'clean versions:set -DnewVersion={} versions:commit ;'
                'clean {} -Pproduction'.format(self.config[pack].get('version', '1.0-SNAPSHOT'), goal),
                'tests/pom.xml': 'clean versions:set -DnewVersion={} versions:commit ;'
                'clean {} -Pproduction,upload-aqua'.format(self.config[pack].get('version', '1.0-SNAPSHOT'), goal),
            },
        )
        self.Context.task_ids.append(task.id)

    def _build_swagger_api_client(self, pack, goal):
        task = create_subtasks.create_maven_build_subtask(
            self,
            self.source_urls[pack],
            {
                'pom.xml': 'clean versions:set -DnewVersion={} versions:commit ;'
                '-Dswagger.api.url={} {}'.format(
                    self.config[pack].get('version', '1.0-SNAPSHOT'), self.config[pack]['url'], goal
                ),
            },
        )
        self.Context.task_ids.append(task.id)

    def _build_direct_db_models(self, pack, goal):
        task = create_subtasks.create_maven_build_subtask(
            self,
            self.source_urls[pack],
            {
                'pom.xml': 'versions:set -DnewVersion=1.0-{db_stage}-SNAPSHOT versions:commit ;'
                'clean {goal} -P{db_stage}'.format(db_stage=self.Parameters.db_stage, goal=goal)
            },
        )
        self.Context.task_ids.append(task.id)

    def _build_regular(self, pack, goal):
        task = create_subtasks.create_maven_build_subtask(
            self,
            self.source_urls[pack],
            {
                'pom.xml': 'clean versions:set -DnewVersion={} versions:commit ;'
                '-Pupload-aqua {}'.format(self.config[pack].get('version', '1.0-SNAPSHOT'), goal)
            },
        )
        self.Context.task_ids.append(task.id)
