# -*- coding: utf-8 -*-

import logging
from collections import defaultdict
from copy import copy

logger = logging.getLogger(__name__)

ROOT = 'ROOT'


def _make_dependency_tree_and_goals(config, tree, goals, pack, rebuild_downstream):
    if pack not in config:
        raise RuntimeError("No config for pack {} found in config file.".format(pack))
    if not config[pack].get('enabled') or config[pack].get('skip'):
        logger.warn("Pack {} is disabled in config".format(pack))
        return

    downstream = config[pack].get('downstream')
    if rebuild_downstream and downstream:
        children_with_goals = {
            child: value['goal'] for key, value in downstream.items() for child in key.split(',') if value.get('goal')
        }
        for child, goal in children_with_goals.items():
            # goal может иметь значение либо 'deploy', либо 'compile'.
            # Если имеются зависимости и 'deploy', и 'compile', то берём 'deploy'.
            if child not in goals or goals[child] != 'deploy':
                goals[child] = goal
        tree[pack] = set(children_with_goals.keys())
        for child in tree[pack]:
            if child not in tree:
                _make_dependency_tree_and_goals(config, tree, goals, child, rebuild_downstream)
    else:
        if not rebuild_downstream:
            logger.info('Downstream is not built due to parameter "rebuild_downstream"')
        tree[pack] = {}


def _remove_circles(tree, parents, pack):
    parents.add(pack)
    for child in copy(tree[pack]):
        if child in parents:
            tree[pack].remove(child)
            logger.info("Remove child {} from pack {} because of circular dependency.".format(child, pack))
        else:
            _remove_circles(tree, parents, child)
    parents.remove(pack)


def _invert_tree(tree):
    inverted_tree = defaultdict(set)
    for pack, children in tree.items():
        inverted_tree[pack]  # create item with empty set if no parents exist
        for child in children:
            inverted_tree[child].add(pack)
    return inverted_tree


def _make_schedule(packs, goals):
    queued_packs = set()
    queue = []
    while packs:
        ready_packs = []
        for pack in copy(packs):
            for child in copy(packs[pack]):
                if child in queued_packs:
                    packs[pack].remove(child)
            if not packs[pack]:
                ready_packs.append((pack, goals[pack]))
                packs.pop(pack)
        queued_packs |= {pack for pack, _ in ready_packs}
        queue.append(ready_packs)
    return queue


def resolve_dependencies(config, first_packs, first_goal, rebuild_downstream):
    tree = {ROOT: set(first_packs)}
    goals = {first: first_goal for first in [ROOT] + first_packs}
    for first in first_packs:
        if first not in tree:
            _make_dependency_tree_and_goals(config, tree, goals, first, rebuild_downstream)
    _remove_circles(tree, set(), ROOT)
    inverted_tree = _invert_tree(tree)
    return _make_schedule(inverted_tree, goals)[1:]
