# -*- coding: utf-8 -*-
import subprocess
from re import sub
from sandbox import sdk2
from sandbox.projects.common.vcs.arc import Arc


def build_feature_regexp(feature):
    camel_case_name = sub(r'_', " ", feature).title().replace(" ", "")
    exclude_import = "^([^i]|i[^m]|im[^p]|imp[^o]|impo[^r]|impor[^t])"
    feature_names = "(" + feature + '|' + ''.join([camel_case_name[0].lower(), camel_case_name[1:]]) + ")"

    return "\"" + exclude_import + ".*" + feature_names + ".*$\""


def recursive_grep(root, mount_point, substring):
    root_regexp = "\"^" + root + ".*\""
    output = subprocess.check_output(
        f"{mount_point}/ya tool cs -i {substring} -f {root_regexp} -m 100", shell=True
    )
    return [
        split_result(bytes.decode(lines))
        for lines in output.splitlines()
    ]


def split_result(result):
    path, line_number, substr = result.split(':', 2)
    return {'path': path, 'line_number': line_number, 'substr': substr}


class DirectCountFeaturesInCode(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        root_paths = sdk2.parameters.String("Root paths to count features from (comma separated)", required=True,
                                            default_value="direct,adv/frontend")
        yt_cluster_name = sdk2.parameters.String("YT cluster", required=True, default_value="hahn")
        yt_input_table = sdk2.parameters.String("YT table to read features from", required=True,
                                                default_value="//home/direct/mysql-sync/current/ppcdict/straight/features")
        yt_output_table = sdk2.parameters.String("YT table to write to", required=True,
                                                 default_value="//home/direct/test/features/features_using_in_code")

        yt_token_secret = sdk2.parameters.YavSecretWithKey(
            'Yt token secret',
            default='sec-01g6n8j7g7phxp6e8t5g8gfftw@ver-01g6n8j7gfkddbds4434mrjg43#yt_robot-direct-feature',
            required=True,
        )

        arc_oauth_token_secret = sdk2.parameters.YavSecretWithKey(
            'Arc oauth token secret',
            default='sec-01fw10pzf0xv4wqh4efnyr8pfv@ver-01fw10pzfyrxdxxbe9j64qb4y3#arc-oauth_robot-direct-feature',
            required=True,
        )

    def on_execute(self):
        yt_token = self.Parameters.yt_token_secret.value()
        arc_oauth_token = self.Parameters.arc_oauth_token_secret.value()
        import yt.wrapper as yt

        client = yt.YtClient(proxy=self.Parameters.yt_cluster_name, token=yt_token)

        feature_list = []
        for row in client.read_table(self.Parameters.yt_input_table):
            feature_list.append(row['feature_text_id'])

        feature_stat_by_name = []
        arc = Arc(arc_oauth_token=arc_oauth_token)

        roots = str(self.Parameters.root_paths).split(",")

        with arc.mount_path("", changeset='trunk', fetch_all=False) as mount_point:
            for feature in feature_list:
                feature_regexp = build_feature_regexp(feature)
                stat = []
                for root in roots:
                    stat += recursive_grep(root, mount_point, feature_regexp)
                feature_stat_by_name.append({'feature': feature, 'lines': stat, "total": len(stat)})

        client.write_table(self.Parameters.yt_output_table, feature_stat_by_name)
