import logging
import re
import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.task import Semaphores
from sandbox.projects.common import binary_task
from sandbox.projects.common.vcs.arc import Arc


class DirectFeatureUpdate(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name='DIRECT_FEATURE_UPDATE_QUEUE'),
            ],
        )

    class Parameters(sdk2.Task.Parameters):
        owner = "DIRECT"
        description = "Update feature file and generate enums"

        ticket = sdk2.parameters.StrictString(
            'Startrek ticket',
            required=True,
            regexp=r'^[A-Z]+-\d+$',
        )

        with sdk2.parameters.RadioGroup('Task mode', required=True) as task_mode:
            task_mode.values.ADD = task_mode.Value('ADD', default=True)
            task_mode.values.DELETE = task_mode.Value('DELETE')

        feature_name = sdk2.parameters.StrictString(
            'Feature name',
            required=True,
            regexp=r'^[0-9a-z_]+$',
        )

        with task_mode.value.ADD:
            feature_description = sdk2.parameters.String('Feature description', required=True)

            with sdk2.parameters.RadioGroup('Feature type', required=True) as feature_type:
                feature_type.values.TEMP = feature_type.Value('TEMP', default=True)
                feature_type.values.PERMANENT = feature_type.Value('PERMANENT')

            deprecated = sdk2.parameters.Bool('Is feature deprecated', default=False)

            enabled_in_e2e = sdk2.parameters.Bool('Is feature enabled in e2e', default=False)

            skip_checks = sdk2.parameters.Bool('Commit without review', default=False)

        with sdk2.parameters.Group('Technical parameters', collapse=True) as technical_parameters:
            arc_user = sdk2.parameters.String('Arc user', default='robot-direct-feature', required=True)

            arc_oauth_token_secret = sdk2.parameters.YavSecretWithKey(
                'Arc oauth token secret',
                default='sec-01fw10pzf0xv4wqh4efnyr8pfv@ver-01fw10pzfyrxdxxbe9j64qb4y3#arc-oauth_robot-direct-feature',
                required=True,
            )

            generate_script_path = sdk2.parameters.String(
                'Path to generate_features.sh',
                default='direct/bin/generate_features.sh',
                required=True,
            )

            config_file_path = sdk2.parameters.String(
                'Path to feature yaml config',
                default='direct/libs-internal/base-model/src/main/resources/feature.yaml',
                required=True,
            )

            _lbrp = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(DirectFeatureUpdate, self).on_execute()

        arc_oauth_token = self.Parameters.arc_oauth_token_secret.value()
        arc = Arc(arc_oauth_token=arc_oauth_token)

        if self.Parameters.ticket == 'DIRECT-163529':
            raise TaskError('Please specify a new ticket for feature commit instead of using the default one (DIRECT-163529)')

        with arc.mount_path("", changeset='trunk', fetch_all=False) as mount_point:
            from direct.infra.feature_generate.lib.feature import read_feature_config, generate_feature_config

            config_file = os.path.join(mount_point, self.Parameters.config_file_path)
            features = read_feature_config(config_file)

            change_description = self.update_features(features)
            message = ("{}: {}\n" "Author: {}\n" "Sandbox task: https://sandbox.yandex-team.ru/task/{}/view").format(
                self.Parameters.ticket,
                change_description,
                self.author,
                self.id,
            )

            with open(config_file, 'w') as f:
                f.write(generate_feature_config(features))

            with sdk2.helpers.ProcessLog(self, logger='generate_features.sh') as processlog:
                sdk2.helpers.subprocess.check_call(
                    [self.Parameters.generate_script_path],
                    stdout=processlog.stdout,
                    stderr=processlog.stderr,
                    cwd=mount_point,
                )

            logging.info('Arc status: {}'.format(arc.status(mount_point)))

            ya_bin = os.path.join(mount_point, 'ya')
            ya_env = dict(YA_USER=self.Parameters.arc_user, YA_TOKEN=arc_oauth_token)

            review_url = self.do_commit(arc, mount_point, ya_bin, ya_env, message)

            if self.Parameters.skip_checks:
                self.force_merge(arc, mount_point, ya_bin, ya_env)

            self.set_info(
                info='Updated feature files: <a href="{}">{}</a>'.format(review_url, review_url),
                do_escape=False,
            )

    def update_features(self, features):
        from direct.infra.feature_generate.lib.feature import Feature

        existing_feature = None
        for feature in features:
            if feature.name.lower() == self.Parameters.feature_name:
                existing_feature = feature
                break

        if self.Parameters.task_mode == 'ADD':
            if existing_feature is not None:
                raise TaskError(
                    'Feature "{}" already exists in feature config'.format(
                        self.Parameters.feature_name,
                    )
                )

            features.append(
                Feature(
                    name=self.Parameters.feature_name,
                    description=self.Parameters.feature_description,
                    type=self.Parameters.feature_type,
                    deprecated=self.Parameters.deprecated,
                    enabled_in_e2e=self.Parameters.enabled_in_e2e,
                )
            )

            return "Add feature " + self.Parameters.feature_name

        elif self.Parameters.task_mode == 'DELETE':
            if existing_feature is None:
                raise TaskError(
                    'Feature "{}" was not found in feature config'.format(
                        self.Parameters.feature_name,
                    )
                )

            features.remove(existing_feature)

            return "Remove feature " + self.Parameters.feature_name

    def do_commit(self, arc, mount_point, ya_bin, ya_env, message):
        branch_name = 'update_feature_{}_{}'.format(self.Parameters.feature_name, self.id)
        arc.checkout(mount_point, branch=branch_name, create_branch=True)

        arc.add(mount_point, all_changes=True)
        arc.commit(mount_point, message, all_changes=True)

        upstream_branch_name = 'users/{}/{}'.format(self.Parameters.arc_user, branch_name)
        arc.push(mount_point, upstream_branch_name)

        with sdk2.helpers.ProcessLog(self, logger='ya_pr_create'):
            pr_output = sdk2.helpers.subprocess.check_output(
                [ya_bin, 'pr', 'create', '--push', '--publish', '--wait', '--message', message],
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=mount_point,
                env=ya_env,
            )

        review_url = re.findall(r'https://a.yandex-team.ru/review/\d+', pr_output)
        if review_url:
            return review_url[0]

        raise Exception('No review url in pr output')

    def force_merge(self, arc, mount_point, ya_bin, ya_env):
        with sdk2.helpers.ProcessLog(self, logger='ya_pr_merge') as processlog:
            sdk2.helpers.subprocess.check_call(
                [ya_bin, 'pr', 'merge', '--now', '--force', '--wait'],
                stdout=processlog.stdout,
                stderr=processlog.stderr,
                cwd=mount_point,
                env=ya_env,
            )
