# -*- encoding: utf8 -*-

import logging
import os

import sandbox.common.types.misc as common_types
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sdk2.helpers import subprocess

# Ресурс из которого запускается LXC-окружение задачи (node.js v16.13.2)
# https://sandbox.yandex-team.ru/task/1248223238/view
LXC_CONTAINER_ID = 2886539959

ROBOT_TWILIGHT_NAME = 'robot-twilight'
YAV_ROBOT_TWILIGHT_SECRET = 'sec-01f49pn04gd2xyk9xvwgnvmh38'
RELATIVE_PROJECT_PATH = os.path.join('adv', 'frontend', 'services', 'uac')


class UacDeployCleanup(sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = common_types.DnsType.DNS64
        container_resource = LXC_CONTAINER_ID
        environments = [environments.GCCEnvironment('5.3.0'), environments.ArcEnvironment()]

    class Parameters(sdk2.Task.Parameters):
        owner = "DIRECT-TEST"
        description = "Remove unused betas"

        secret_data = sdk2.parameters.YavSecret(
            label='ID секрета в Секретнице',
            description='Обязательные поля: dctl-yp-token, awacs-oauth-token, arc-token, ya-token',
            default_value=YAV_ROBOT_TWILIGHT_SECRET,
            required=True,
        )

    def _prepare_env(self):
        for env in self.Requirements.environments:
            env.prepare()

    def _get_yav_secret_data(self):
        return self.Parameters.secret_data.data()

    def _get_arcanum_oauth_token(self):
        secret_data = self._get_yav_secret_data()
        arcanum_oauth_token = secret_data['arc-token']

        return arcanum_oauth_token

    def _get_dctl_oauth_token(self):
        secret_data = self._get_yav_secret_data()
        dctl_oauth_token = secret_data['dctl-yp-token']

        return dctl_oauth_token

    def _get_awacs_oauth_token(self):
        secret_data = self._get_yav_secret_data()
        awacs_oauth_token = secret_data['awacs-oauth-token']

        return awacs_oauth_token

    def _get_ya_token(self):
        secret_data = self._get_yav_secret_data()
        ya_token = secret_data['ya-token']

        return ya_token

    def _install_npm_dependencies(self, project_path):
        logging.info("Dependencies installation started")

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            subprocess.check_call(
                ['pnpm', 'run', 'install:deps'],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            )

        logging.info("Dependencies installation finished")

    def _deploy_cleanup(self, project_path, ya_bin_path):
        logging.info("Deploy cleanup started")

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            env = os.environ

            env["ARCANUM_OAUTH_TOKEN"] = self._get_arcanum_oauth_token()
            env["AWACS_OAUTH_TOKEN"] = self._get_awacs_oauth_token()
            env["DCTL_YP_TOKEN"] = self._get_dctl_oauth_token()
            env["YA_TOKEN"] = self._get_ya_token()
            env["YA_USER"] = ROBOT_TWILIGHT_NAME
            env["LOGNAME"] = ROBOT_TWILIGHT_NAME
            env["YA_BIN"] = ya_bin_path

            subprocess.check_call(
                ['pnpm', 'run', 'ci:ydeploy:cleanup'],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=env,
            )

        logging.info("Deploy cleanup finished")

    def on_execute(self):
        self._prepare_env()

        arcanum_oauth_token = self._get_arcanum_oauth_token()

        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk', arc_oauth_token=arcanum_oauth_token) as arcadia_root:
            project_path = os.path.join(arcadia_root, RELATIVE_PROJECT_PATH)
            ya_bin_path = os.path.join(arcadia_root, 'ya')

            logging.info('Changing path to {}'.format(project_path))
            os.chdir(project_path)

            self._install_npm_dependencies(project_path)
            self._deploy_cleanup(project_path, ya_bin_path)
