# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import logging
from operator import itemgetter

from .base import BaseExecutor
from ..target_types.base import ClickHouseInsertTarget

logger = logging.getLogger(__name__)


class ClickHouseInsertExecutor(BaseExecutor):
    target_class = ClickHouseInsertTarget
    _process_funcs = {
        'string': lambda x: "'{}'".format(x.replace('\n', '\\n').replace('\t', '\\t').replace("'", "\\'")),
        'integer': repr,
        'double': repr,
        'bool': lambda x: '1' if x else '0',
    }
    _none_consts = {
        'string': "''",
        'integer': "0",
        'double': "0.0",
        'bool': '0',
    }
    _chunk_size = 2000
    _query_base_template = 'INSERT INTO {src_database}.{src_table} ({src_columns}) VALUES'

    def __init__(self, *args, **kwargs):
        super(ClickHouseInsertExecutor, self).__init__(*args, **kwargs)
        self._query_template = self.prepare_template()
        self._query = self._query_template
        self._rows = []

    def prepare_template(self):
        query = self._query_base_template.format(src_database=self.database, src_table=self.table,
                                                 src_columns=', '.join(map(itemgetter(0), self.columns)))
        return query + ' {values};'

    def is_valid(self):
        return len(self._target.dependencies) == 1

    def execute(self, transaction_id=None):
        source = self._target.dependencies[0]
        for row in self.yt.read_table(self.get_output_path(source),
                                      format=self.yt.JsonFormat(attributes={"encode_utf8": False})):
            self.process_row(row)
        self.run_query()

        self.yt_write_table(self.output_path)

    def process_row(self, row):
        values = []
        for name, desc in self.columns:
            values.append(self.to_string(row.get(name), desc))

        self._rows.append('({})'.format(', '.join(values)))

        if len(self._rows) > self._chunk_size:
            self.run_query()

    def to_string(self, value, desc):
        ftype = desc.get('type')
        if ftype in self._process_funcs:
            if value is None:
                return self._none_consts[ftype]

            return self._process_funcs[ftype](value)
        elif ftype == 'context':
            return "'{}'".format(self._context[desc['context']])
        else:
            raise RuntimeError('Wrong config: %s', desc)

    def run_query(self):
        if not self._rows:
            return
        query_text = self._query.format(values=', '.join(self._rows))
        logger.info('Will run query: %s', query_text)
        r = self.clickhouse.query(query_text.encode('utf-8'), timeout=(4, 600))
        logger.info(b'Response is %s', r.text)
        self._query = self._query_template
        self._rows = []

    @property
    def clickhouse(self):
        return self._config['clickhouse_client']

    @property
    def columns(self):
        # noinspection PyUnresolvedReferences
        return self._target.columns

    @property
    def database(self):
        # noinspection PyUnresolvedReferences
        return self._target.database

    @property
    def table(self):
        # noinspection PyUnresolvedReferences
        return self._target.table
