#!/usr/bin/env python
# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function, division

import argparse
import logging
import sys
import time
from ConfigParser import SafeConfigParser
from datetime import datetime
from multiprocessing import Process, Manager, Lock, Value

from sandbox.projects.direct_internal_analytics.laborer.executor import get_executor
from sandbox.projects.direct_internal_analytics.laborer_base import const
from sandbox.projects.direct_internal_analytics.laborer_base.context import deserialize_context, get_context
from sandbox.projects.direct_internal_analytics.laborer_base.exceptions import TargetAlreadyProcessedException
from sandbox.projects.direct_internal_analytics.laborer_base.imports import get_target_by_name
from sandbox.projects.direct_internal_analytics.laborer_base.processing import process_target_new, \
    TargetProcessorFactory, TargetProcessor


def main():
    pars = argparse.ArgumentParser(description="Run yql query")
    pars.add_argument(const.YT_CLUSTER, help="YT Cluster", required=True)

    pars.add_argument(const.CH_HOST, help="ClickHouse Host", required=True)
    pars.add_argument(const.CH_USER, help="ClickHouse User", required=True)

    pars.add_argument(const.TVM_ID, help="TVM app ID")

    pars.add_argument(const.CONFIG, help="Tokens config file")

    pars.add_argument(const.CH_PWD, help="ClickHouse Password")
    pars.add_argument(const.YQL_TOKEN, help="YQL Token")
    pars.add_argument(const.YT_TOKEN, help="YT Token")
    pars.add_argument(const.TVM_SECRET, help="TVM app secret")
    pars.add_argument(const.APPSFLYER_API_TOKEN, help="appsflyer api token secret")

    pars.add_argument(const.DESTINATION, help="YT destination folder for result table", required=True)
    pars.add_argument(const.TARGET, help="Calculation target", required=True)

    pars.add_argument(const.FORCE, action='store_true', help="Force rebuild target")
    pars.add_argument(const.WITH_DEPS, action='store_true', help="Build all dependencies before building target")
    pars.add_argument(const.DO_NOT_DELETE_TMP, action='store_true',
                      help="Don't delete temporary data even if target is final")
    pars.add_argument(const.CLEAN_UP_ONLY, action='store_true',
                      help="Don't process target, only delete it's temporary dependencies")

    pars.add_argument(const.CONTEXT, help="Execution context")
    pars.add_argument(const.DATE, help="Context date (rewrites context value)")
    pars.add_argument(const.TOKEN, help="Context token (rewrites context value)")
    pars.add_argument(const.UNPACKED_TARGET_PATH, help="Path to unpacked resource contents")
    opts = pars.parse_args()

    if opts.config:
        cp = SafeConfigParser()
        cp.read((opts.config,))
        yt_token = cp.get(const.C_SECTION, const.C_YT_TOKEN)
        yql_token = cp.get(const.C_SECTION, const.C_YQL_TOKEN)
        ch_pwd = None
        if cp.has_option(const.C_SECTION, const.C_CH_PWD):
            ch_pwd = cp.get(const.C_SECTION, const.C_CH_PWD)
        tvm_secret = None
        if cp.has_option(const.C_SECTION, const.C_TVM_SECRET):
            tvm_secret = cp.get(const.C_SECTION, const.C_TVM_SECRET)
        appsflyer_api_token = None
        if cp.has_option(const.C_SECTION, const.C_APPSFLYER_API_TOKEN):
            appsflyer_api_token = cp.get(const.C_SECTION, const.C_APPSFLYER_API_TOKEN)
    else:
        yt_token = opts.yt_token
        yql_token = opts.yql_token
        ch_pwd = opts.ch_pwd
        tvm_secret = opts.tvm_secret
        appsflyer_api_token = opts.appsflyer_api_token

    context = deserialize_context(opts.context) if opts.context else get_context()
    if opts.date:
        context['date'] = datetime.strptime(opts.date, '%Y-%m-%d').date()
    if opts.token:
        context['token'] = opts.token
    if opts.unpacked_target_path:
        context['unpacked_target_path'] = opts.unpacked_target_path

    run_processing(opts.target, opts.yt_cluster, opts.destination, yt_token, yql_token, opts.ch_host, opts.ch_user,
                   ch_pwd, opts.tvm_id, tvm_secret, appsflyer_api_token, context, opts.force, opts.with_dependencies, opts.do_not_delete_temporary,
                   opts.clean_up_only)


class ErrorMarker(object):
    def __init__(self):
        self._error = False
        self._lock = Lock()

    def set_error(self):
        with self._lock:
            self._error = True

    def has_error(self):
        with self._lock:
            return self._error


def run_processing(target, yt_cluster, destination, yt_token, yql_token, ch_host, ch_user, ch_pwd, tvm_id, tvm_secret, appsflyer_api_token,
                   context, force=False, with_dependencies=True, do_not_delete_temporary=False, clean_up_only=False):
    logging.info('Execution context: %s', context)

    config_kwargs = {'yt_cluster': yt_cluster, 'destination': destination, 'yt_token': yt_token,
                     'yql_token': yql_token, 'ch_host': ch_host, 'ch_user': ch_user, 'ch_pwd': ch_pwd,
                     'tvm_id': tvm_id, 'tvm_secret': tvm_secret, 'appsflyer_api_token': appsflyer_api_token}
    factory = ThreadingTargetProcessorFactory(config_kwargs, context, force)

    target_cls = get_target_by_name(target)
    process_target_new(target_cls, factory, with_dependencies=with_dependencies, force=force,
                       clean_up_only=clean_up_only)
    factory.join()

    if with_dependencies and target_cls.final and not do_not_delete_temporary:
        factory.delete_non_final()


def get_config(yt_cluster, destination, yt_token, yql_token, ch_host, ch_user, ch_pwd, tvm_id, tvm_secret, appsflyer_api_token):
    import yt.wrapper as yt
    from clickhouse.database import Database
    from yql.api.v1.client import YqlClient

    if not yt_token or not yql_token:
        raise RuntimeError('You must provide tokens through config file or commandline parameters')
    yt.config["proxy"]["url"] = "{}.yt.yandex.net".format(yt_cluster)
    yt.config['token'] = yt_token
    yql = YqlClient(db=yt_cluster, token=yql_token)
    if ch_user:
        ch = Database(ch_host, None, username=ch_user, password=ch_pwd)
    else:
        ch = Database(ch_host, None)

    return {
        'yt_client': yt,
        'yql_client': yql,
        'clickhouse_client': ch,
        'home': destination,
        'tvm_id': tvm_id,
        'tvm_secret': tvm_secret,
        'appsflyer_api_token': appsflyer_api_token,
    }


def create_executor(target, config_kwargs, context, force):
    return get_executor(
        target,
        get_config(**config_kwargs),
        context,
        force=force
    )


class ThreadingTargetProcessorFactory(TargetProcessorFactory):
    def __init__(self, config_kwargs, context, force):
        self._config_kwargs = config_kwargs
        self._context = context
        self._force = force
        self._error_manager = Value('i')
        self._error_manager.value = 0
        self._process_storage = []
        self._ttp_storage = {}

    def get_processor_for(self, target):
        ttp = ThreadingTargetProcessor(target, self._config_kwargs, self._context, self._force, self._error_manager,
                                       self._process_storage)
        self._ttp_storage[target] = ttp
        return ttp

    def join(self):
        for p in self._process_storage:
            p.join()

        if self._error_manager.value:
            raise RuntimeError('Got an error while executing code')

    def delete_non_final(self):
        for target in self._ttp_storage:
            if target.final:
                continue

            self._ttp_storage[target].clear_data()


class ThreadingTargetProcessor(TargetProcessor):
    def __init__(self, target, config_kwargs, context, force, error_manager, process_storage):
        self._config_kwargs = config_kwargs
        self._executor = create_executor(target, config_kwargs, context, force)
        self._lock = Lock()
        self._target = target
        self._context = context
        self._force = force
        self._error_manager = error_manager
        self._process_storage = process_storage

    def clear_data(self):
        name = get_target_name(self._target)
        logging.info('Deleting data for %s', name)
        self._executor.delete_data()

    def get_lock(self):
        return self._lock

    def run_processing(self, dependencies_locks):
        name = get_target_name(self._target)

        ready_manager = Manager().dict()
        ready_manager[name] = False

        p = Process(target=run_executor,
                    args=(self._target, self._config_kwargs, self._context, self._force, self._lock, dependencies_locks,
                          ready_manager, self._error_manager),
                    name=name)
        logging.info('Start thread for %s', name)
        p.start()

        self._process_storage.append(p)

        while not ready_manager[name]:
            logging.info('Waiting lock %s to aquire', name)
            time.sleep(0.1)

    def clear_locks(self):
        pass

    def is_already_processed(self):
        return self._executor.yt_node_exists()


def get_target_name(target):
    return '{}.{}'.format(target.__module__, target.__name__)


def run_executor(target, config_kwargs, context, force, lock, dependencies_locks, ready_manager, error_manager):
    with lock:
        name = get_target_name(target)
        ready_manager[name] = True
        executor = create_executor(
            target,
            config_kwargs,
            context,
            force
        )

        for d_lock in dependencies_locks:
            logging.info("dep check lock %s", d_lock)
            with d_lock:
                logging.info("dep ready %s", d_lock)

        if error_manager.value:
            logging.info("got error, returning")
            return

        try:
            executor.start_execution()
        except TargetAlreadyProcessedException:
            logging.info("Target already processed")
        except Exception as e:
            logging.exception(e)
            with error_manager.get_lock():
                error_manager.value = 1

        logging.info("release %s", lock)


if __name__ == '__main__':
    logging.basicConfig(stream=sys.stdout, level=logging.INFO,
                        format="%(asctime)s\t%(levelname)s\t%(name)s\t%(threadName)s\t%(message)s")
    main()
