# coding=utf-8
"""Модуль, содержащии типовые базовые классы для описания целей. Для описания цели нужно отнаследоваться от типового
(не BaseTarget) наследника базового класса и обязательно переопределить все неопределенные (None) значения по всей
цепочке наследования. При необходимости можно переопределить поля с уже заданными пустыми значениями, например
зависимости цели (dependencies)

Пример:

>>> from projects.direct_internal_analytics.laborer.target_types.base import ClickHouseSelectTarget, YqlSelectTarget
>>>
>>> class FirstTarget(ClickHouseSelectTarget):
>>>     title = 'first_target'
>>>     query = '''SELECT 1 FROM my_first_table;'''
>>>     columns = ...
>>>     columns_types = ...
>>>
>>> class SecondTarget(YqlSelectTarget):
>>>     title = "second_target"
>>>     query = '''SELECT 1 FROM my_second_table;'''
>>>     dependencies = [FirstTarget]

Другие примеры можно посмотреть в модуле `projects.direct_internal_analytics.laborer.targets`
"""
from __future__ import absolute_import, unicode_literals, print_function


class BaseTarget(object):
    """Базовый объект описания цели для сборки, содержит основные параметры,
    необходимые для определения того, что мы хотим сделать
    """
    # Модификатор имени таргета при создании путей для результатов. Можно перезадать в базовых склассах и не стоит
    # перезадавать в описании конкретных целей
    _namespace = ""

    # Шаблон ноды кипариса, которая соответсвует результату работы таргета
    target_template_path = None
    # Время жизни результата в днях (по умолчанию - без ограничений)
    expiration_days = None

    # Название цели. Должно соответствовать регекспу r'^[a-zA-Z0-9_]+$'
    title = None
    # Список классов зависимостей
    dependencies = []
    # Является ли цель конечной в своем графе
    final = False
    # Выполнение таргета по-умолчанию выполняется внути внешней транзакции, но можно отключить
    execute_in_transaction = True


class YqlSelectTarget(BaseTarget):
    """Читающий запрос в YQL, сохраняет результат в YT-таблицу"""
    _namespace = "yql_"

    # Текст запроса в YQL. Обработчик этого типа цели не умеет обрабатывать данные из результатов запроса, поэтому
    # результаты запроса нужно сохранять с помощью INSERT INTO [{insert_target}]
    # Для использования в теле запроса доступно несколько плейсхолдеров:
    # - {insert_target} - таблица, в которую нужно вставить результат
    # - {date} - дата, за которую делается отчет. Можно использовать в синтаксисе .format, например как {date:%Y%m} для
    #            получения только года и месяца
    # - {dependencies} - список строк с адресами таблиц результатов зависимостей в том порядке, в котором они
    #                    перечислены в аттрибуте класса с описанием. Можно использовать как {dependencies[0]} для
    #                    получения конкретного адреса
    query = None
    # Specify YQL syntax version for this query: 0 (old, default) or 1 (new)
    yql_syntax_version = 0

    @classmethod
    def user_attributes(cls, context):
        return {}


class ClickHouseSelectTarget(BaseTarget):
    """Читающий запрос в ClickHouse, сохраняет результат в YT-таблицу"""
    _namespace = "ch_"

    # Текст запроса в ClickHouse. Обработчик цели работает с результатом, который вернул запрос. Для подстановки можно
    # использовать плеисхолдер {date}, как у YqlSelectTarget
    query = None
    # Список колонок выдачи в том порядке, в котором они указаны в запросе
    columns = None
    # Словарь с соответствием типа колонки типу ее данных. Сейчас есть три типа: "string", "integer", "double"
    columns_types = None


class ClickHouseInsertTarget(BaseTarget):
    """Запрос, пишущий в ClickHouse все данные из YT-таблицы"""
    _namespace = "ch_wr_"

    # Название базы данных, в которую нужно загрузить данные
    database = None
    # Название таблицы в базе данных, в которую нужно загрузить данные
    table = None
    # Список названий колонок и их описаний, пример можно посмотреть в существующих наследниках
    columns = None
