# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import base64
import pickle
import uuid
from datetime import datetime, timedelta, time

import sandbox.common.types.resource as ctr
from sandbox.projects.direct_internal_analytics.laborer_base.processing import get_data_path
from sandbox.projects.direct_internal_analytics.tasks.resources import AdAnalyticsRunnerSyncResource


def get_curdate(date_shift=None, date_step=None, date_auto_refill_shift=None, token=None, params=None, target=None, home=None, yt_cluster=None, is_clean=None):

    shift_count = 0
    dt = datetime.combine(datetime.now().date(), time(0, 0))
    while dt + timedelta(hours=date_step * shift_count) < datetime.now():
        shift_count = shift_count + 1

    now = dt + timedelta(hours=date_step * (shift_count - 1))

    orig_date_shift = date_shift
    clean_date = None

    while date_auto_refill_shift >= date_shift:
        clean_date = (now - timedelta(hours=date_shift))
        resource = AdAnalyticsRunnerSyncResource.find(
            state=ctr.State.READY,
            attrs={'data_path':
                       get_data_path(target, {
                           'timestamp': now.isoformat(),
                           'date': clean_date,
                           'token': token or str(uuid.uuid4())[:8],
                           'params': params or {},
                           'home': home,
                           'yt_cluster': yt_cluster,
                       })
                   }
        ).first()

        if resource is not None:
            break

        date_shift = date_shift + date_step

    if date_shift != orig_date_shift:
        date_shift = date_shift - date_step

    return {
        'date': (now - timedelta(hours=date_shift)),
        'clean_date': clean_date
    }


def get_context(
        token=None,
        date=None,
        params=None,
        date_shift=None,
        date_step=None,
        date_auto_refill_shift=None,
        target=None,
        expiration_days=None,
        yt_cluster=None,
        home=None):
    """Получить словарь со стандартным контекстом исполнения расчета

    :param token: токен для контекста, если не задан, используется случайное значение
    :type token: basestring
    :param date: дата для контектса, если не задана, используется день, предшествующий текущему
    :type date: datetime.date
    :param params: дата для контектса, если не задана, используется день, предшествующий текущему
    :type params: dict[basestring]
    :param expiration_days: сколько дней должен жить результат
    :type expiration_days: int

    :rtype: dict[basestring]
    """
    now = datetime.now()

    current_date = None
    clean_date = None

    if date:
        current_date = date
        clean_date = date
    else:
        dates = get_curdate(
            date_shift=date_shift,
            date_step=date_step,
            date_auto_refill_shift=date_auto_refill_shift,
            token=token,
            params=params,
            target=target,
            home=home,
            yt_cluster=yt_cluster,
        )

        current_date = dates['date']
        clean_date = dates['clean_date']

    return {
        'timestamp': now.isoformat(),
        'date': current_date,
        'token': token or str(uuid.uuid4())[:8],
        'params': params or {},
        'clean_date': clean_date,
        'expiration_days': expiration_days,
        'yt_cluster': yt_cluster
    }


def serialize_context(context):
    """Сериализовать контекст в виде строки"""
    return base64.b64encode(pickle.dumps(context))


def deserialize_context(string):
    """Десериализовать контекст из строки"""
    return pickle.loads(base64.b64decode(string))
