# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import importlib
import os
import sys
import tarfile
import tempfile

from sandbox.projects.direct_internal_analytics.laborer.target_types.base import BaseTarget

SRC_PREFIX = 'src'

base_mod = __name__
parent_mod = '.'.join(base_mod.split('.')[:-1])
content_path = os.path.join(os.path.dirname(os.path.dirname(os.path.realpath(__file__))), 'laborer', 'content')


def get_target_by_name(name):
    """Получить класс цели по ее полному пути импорта"""
    mod, cls = name.rsplit('.', 1)
    mod = importlib.import_module(mod)
    return getattr(mod, cls)


def add_package_to_path(package):
    """Распаковать tar-архив с python-пакетом и добавить его в sys.path

    :param package: полный путь к tar-архиву
    """
    out_path = tempfile.mkdtemp()
    with tarfile.open(package, 'r') as package:
        package.extractall(out_path)

    sys.path.insert(0, os.path.join(out_path, SRC_PREFIX))

    return out_path


_targets = []


def get_all_targets(paths=None, with_intermediate=False):
    global _targets

    if not _targets:
        collect_paths = [content_path]
        if paths:
            collect_paths.extend(paths)
        _targets = collect_targets(collect_paths)
    elif paths:
        raise RuntimeError('targets already collected')

    if with_intermediate:
        return list(_mod_name(x.__module__, x.__name__) for x in _targets)
    else:
        return list(_mod_name(x.__module__, x.__name__) for x in _targets if x.final)


def collect_targets(paths):
    targets = []
    base_target_module = BaseTarget.__module__.rsplit('.', 1)[0]
    for path in paths:
        for obj in get_all_importable_objects(path, get_base_mod(path), superclass=BaseTarget):
            if not obj.__module__.startswith(base_target_module):
                targets.append(obj)

    return list(sorted(set(targets), key=lambda x: _mod_name(x.__module__, x.__name__)))


def get_base_mod(path):
    results = []
    for p in sys.path:
        if path.startswith(p):
            results.append(path[len(p):].strip('/').replace('/', '.'))

    if not results:
        raise RuntimeError()

    return sorted(results, key=lambda x: len(x))[0]


def get_all_importable_objects(base_path, base_module, superclass=None):
    for root, dirs, files in os.walk(base_path, topdown=False):
        root_addition = os.path.relpath(root, base_path)
        if root_addition == "." or not root_addition:
            root_mod = base_module
        else:
            root_mod = _mod_name(base_module, root_addition.replace("/", "."))
        for mfile in files:
            if mfile.endswith(".py") and not mfile.startswith('__init__'):
                top_module = mfile.replace(".py", "")

                mod = importlib.import_module("{}.{}".format(root_mod, top_module))
                for obj in mod.__dict__.values():
                    if superclass is not None:
                        try:
                            if issubclass(obj, superclass):
                                yield obj
                        except TypeError:
                            pass
                    else:
                        yield obj


def _mod_name(base, additional):
    return "{}.{}".format(base, additional)
