# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import logging
import os
from datetime import datetime, timedelta

from sandbox.projects.direct_internal_analytics.laborer_base.exceptions import TargetDependenciesNotReadyException

logger = logging.getLogger(__name__)


def get_data_path(target, context):
    """Получить путь в YT к результату работы над целью с заданным контекстом.

    :type target: Type[projects.direct_internal_analytics.laborer.target_types.base.BaseTarget]
    :type context: dict[basestring]
    """
    title = target.title.replace('/', '_')
    # noinspection PyProtectedMember

    target_template_path = '{home}/{token}/{_namespace}{_title}/{_namespace}{_title}-{date:%Y-%m-%d}'

    if target.target_template_path:
        target_template_path = target.target_template_path

    path = os.path.normpath(target_template_path.format(_namespace=target._namespace, _title=title, **context))
    path = path.strip('/')
    return '//{}'.format(path)


def get_expiration_time(target, context, dt=None):
    if dt is None:
        dt = datetime.utcnow()

    context_expiration_days = context['expiration_days']
    if context_expiration_days:
        context_expiration_days = int(context_expiration_days)
    else:
        context_expiration_days = 0

    target_expiration_days = target.expiration_days
    if target_expiration_days:
        target_expiration_days = int(target_expiration_days)
    else:
        target_expiration_days = 0

    logger.info('Process expiration_time. context_expiration_days:%s, target_expiration_days:%s',
                context['expiration_days'], target.expiration_days)

    if target_expiration_days <= 0 and context_expiration_days <= 0:
        return None

    if target_expiration_days <= 0 or (0 < context_expiration_days <= target_expiration_days):
        return (dt + timedelta(days=context_expiration_days)).isoformat() + 'Z'

    if context_expiration_days <= 0 or (0 < target_expiration_days <= context_expiration_days):
        return (dt + timedelta(days=target_expiration_days)).isoformat() + 'Z'

    return None


class TargetProcessorFactory(object):
    def get_processor_for(self, target):
        """Получить обработчик для заданной цели

        :param target: класс с описанием цели
        :type target: Type[projects.direct_internal_analytics.laborer.target_types.base.BaseTarget]

        :return:
        :rtype: TargetProcessor
        """
        raise NotImplementedError()


class TargetProcessor(object):
    def get_lock(self):
        raise NotImplementedError()

    def is_already_processed(self):
        raise NotImplementedError()

    def run_processing(self, dependencies_locks):
        raise NotImplementedError()

    def clear_locks(self):
        raise NotImplementedError()

    def clear_data(self):
        raise NotImplementedError()


def process_target_new(target, processor_factory, with_dependencies=False, already_processed=None, force=False,
                       clean_up_only=False):
    """Обработать цель с помощью переданного в обработчике исполняемого кода. При необходимости предварительно
    обработать зависимости тем же обработчиком

    :param target: класс с описанием цели
    :type target: Type[projects.direct_internal_analytics.laborer.target_types.base.BaseTarget]
    :param processor_factory: ...
    :type processor_factory: TargetProcessorFactory
    :param with_dependencies: рекурсивно обработать все зависимости цели
    :type with_dependencies: bool
    :param already_processed: словарь, ставящий цели в соответствие результат работы переданной в метод функции
    :type already_processed: dict
    :param force:
    :type force: bool
    :param clean_up_only: Ничего не считать, только удалить промежуточные данные расчета (все, кроме final и первого
                          таргета)
    :type clean_up_only: bool
    """
    first_target = False
    if already_processed is None:
        first_target = True
        already_processed = {}

    logger.info('Preparing to process %s', target)
    if target in already_processed:
        logger.info('%s is already processed', target)
        return

    processor = processor_factory.get_processor_for(target)
    if processor.is_already_processed():
        logger.info('%s is already processed', target)
        if force or (clean_up_only and not target.final and not first_target):
            logger.info('Clearing data and locks of %s to force-restart processing', target)
            processor.clear_data()
            if not clean_up_only:
                processor.clear_locks()
        else:
            already_processed[target] = processor.get_lock()
            if not clean_up_only:
                return

    dependencies_locks = []
    if target.dependencies:
        if with_dependencies or clean_up_only:
            logger.info('Checking dependencies of %s', target)
            for dep in target.dependencies:
                process_target_new(dep, processor_factory, with_dependencies=True, already_processed=already_processed,
                                   force=force, clean_up_only=clean_up_only)

                assert dep in already_processed
                dependencies_locks.append(already_processed[dep])
        else:
            for dep in target.dependencies:
                dep_processor = processor_factory.get_processor_for(dep)
                if dep_processor.is_already_processed():
                    dependencies_locks.append(dep_processor.get_lock())
                else:
                    raise TargetDependenciesNotReadyException()

    if not clean_up_only:
        logger.info('Starting to process %s', target)
        processor.run_processing(dependencies_locks)
    already_processed[target] = processor.get_lock()
