#!/usr/bin/env python
# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function, division

import argparse
import logging
import os
import subprocess
import sys
import time
import tempfile
from ConfigParser import SafeConfigParser
from datetime import datetime, timedelta

from sandbox.projects.direct_internal_analytics.laborer_base import const

TODAY = datetime.now().date()
LOGS_DIR = 'logs'
SYNC_FILE = 'current_date.lock'
PERIODS = (
    ((TODAY - timedelta(days=29)).isoformat(), 'top_partners'),
    ((TODAY - timedelta(days=69)).isoformat(), 'top_partners_retro'),
    ('2017-03-28', 'top_partners_retro_no_rtbdsp'),
    ('2016-10-15', 'top_partners_retro_no_rtbdsp_no_eventtime'),
    ('2015-01-01', 'top_partners_retro_no_rtbdsp_no_bshit'),
)


def get_all_dates(start_date):
    dates = []
    current_date = start_date
    for item_date, item_module in PERIODS:
        period_start_date = datetime.strptime(item_date, '%Y-%m-%d').date()
        while current_date >= period_start_date:
            dates.append((current_date.isoformat(), item_module))
            current_date = current_date - timedelta(days=1)

    return dates


def sync_get(default=None):
    try:
        with open(SYNC_FILE) as f:
            return datetime.strptime(f.read(), '%Y-%m-%d').date() - timedelta(days=1)
    except Exception:
        if default:
            return datetime.strptime(default, '%Y-%m-%d').date()
        return datetime.now().date() - timedelta(days=1)


def sync_set(date_str):
    with open(SYNC_FILE, 'w') as f:
        f.write(date_str)


def calc_target_with_retry(date_str, mod, target, yql_token, yt_token, ch_host, ch_user, ch_pwd, need_deps=True):
    for i in xrange(3):
        try:
            return calc_target(date_str, mod, target, yql_token, yt_token, ch_host, ch_user, ch_pwd, need_deps)
        except RuntimeError as e:
            if i == 2:
                raise e

            sleep_time = 60 * (i + 1)
            logging.error("Got exception, retrying after sleeping %s", sleep_time)
            time.sleep(sleep_time)


def calc_target(date_str, mod, target, yql_token, yt_token, ch_host, ch_user, ch_pwd, need_deps=True):
    cp = SafeConfigParser()
    cp.add_section(const.C_SECTION)
    cp.set(const.C_SECTION, const.C_YQL_TOKEN, yql_token)
    cp.set(const.C_SECTION, const.C_YT_TOKEN, yt_token)
    cp.set(const.C_SECTION, const.C_CH_PWD, ch_pwd)

    script_path = os.path.join(os.path.dirname(os.path.dirname(os.path.realpath(__file__))),
                               'laborer', 'scripts', 'make_target.py')

    with tempfile.NamedTemporaryFile('w', bufsize=0) as cf:
        cp.write(cf.file)
        cf.file.flush()

        process_description = [
            sys.executable, script_path,
            const.YT_CLUSTER, 'hahn',
            const.CH_HOST, ch_host,
            const.CH_USER, ch_user,
            const.CONFIG, cf.name,
            const.DESTINATION, 'home/partner-top',
            const.TOKEN, 'retro',
            const.TARGET, 'projects.direct_internal_analytics.targets.{}.{}'.format(mod, target),
            const.DATE, date_str,
        ]
        if need_deps:
            process_description.append(const.WITH_DEPS)

        env = {'PYTHONPATH': ':'.join(sys.path)}

        logging.info("Launching process %s with env %s", process_description, env)

        if not os.path.exists(LOGS_DIR):
            os.mkdir(LOGS_DIR)
        filename = os.path.join(LOGS_DIR, '{}_{}.'.format(date_str, target))
        with open(filename + 'out', 'w') as out, open(filename + 'err', 'w') as err:
            p = subprocess.Popen(process_description, stdout=out, stderr=err, env=env)
            code = p.wait()

    if code != 0:
        raise RuntimeError("Process exited with code {}. Check logs in {}".format(code, filename))


def calc(date_str, mod, opts):
    calc_target_with_retry(date_str, mod, 'TopPartnersReport', opts.yql, opts.yt, opts.chostd, opts.cuserd, opts.cpwdd)
    calc_target(date_str, mod, 'AprilUpload', opts.yql, opts.yt, opts.chostu, opts.cuseru, opts.cpwdu, False)


def main():
    pars = argparse.ArgumentParser(description="Run yql query")
    pars.add_argument('--yql', help="YQL Token", required=True)
    pars.add_argument('--yt', help="YT Token", required=True)

    pars.add_argument('--chostd', help="ClickHouse Host Download", required=True)
    pars.add_argument('--cuserd', help="ClickHouse User Download", required=True)
    pars.add_argument('--cpwdd', help="ClickHouse Password Download", required=True)

    pars.add_argument('--chostu', help="ClickHouse Host Upload", required=True)
    pars.add_argument('--cuseru', help="ClickHouse User Upload", required=True)
    pars.add_argument('--cpwdu', help="ClickHouse Password Upload", required=True)

    pars.add_argument('--date', help="Start date", required=True)
    opts = pars.parse_args()

    start_date = sync_get(default=opts.date)
    logging.info('Start calculating from %s', start_date)
    dates = get_all_dates(start_date)
    logging.info('Going to calculate %s days', len(dates))
    for num, item in enumerate(dates, 1):
        logging.info('Calculating date [%s/%s]', num, len(dates))
        calc(item[0], item[1], opts)
        logging.info('Finished date [%s/%s]', num, len(dates))
        sync_set(item[0])


if __name__ == '__main__':
    logging.basicConfig(stream=sys.stdout, level=logging.INFO,
                        format="%(asctime)s\t%(levelname)s\t%(name)s\t%(threadName)s\t%(message)s")
    main()
