# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

from collections import OrderedDict

from datetime import timedelta
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import COMMASPACE, formatdate
from smtplib import SMTP

from sandbox.projects.direct_internal_analytics.laborer.target_types.base import YqlSelectTarget
from sandbox.projects.direct_internal_analytics.laborer.target_types.code import ExecuteWithYtTarget


class TopPartnerChanges(YqlSelectTarget):
    title = "top_changes"

    final = True

    query = """
    INSERT INTO [{{ insert_target }}] WITH TRUNCATE
    SELECT
        login,
        device_type,
        target_type,
        current_position,
        old_position,
        position_diff
    FROM (
        SELECT
            NVL(old.login, current.login, "No login") AS login,
            NVL(old.device_type, current.device_type, "No type") AS device_type,
            NVL(old.target_type, current.target_type, "No type") AS target_type,
            current.position AS current_position,
            old.position AS old_position,
            (CAST(old.position AS Int64) - CAST(current.position AS Int64)) AS position_diff
        FROM (
            SELECT
              login,
              device_type,
              target_type,
              event_date,
              position
            FROM [home/partner-top/production/yql_top_partners/yql_top_partners-{{ date|format_date('%Y-%m-%d') }}]
            ) AS current
        FULL JOIN (
            SELECT
              login,
              device_type,
              target_type,
              event_date,
              position
            FROM [home/partner-top/production/yql_top_partners/yql_top_partners-{{ date|minus_days(7)|format_date('%Y-%m-%d') }}]
            ) AS old ON current.login = old.login AND
                    current.device_type = old.device_type AND
                    current.target_type = old.target_type
    )
    WHERE (
            old_position IS NULL OR current_position IS NULL OR position_diff != 0
        ) AND (
            current_position BETWEEN 1 AND 15 OR
            old_position BETWEEN 1 AND 15 OR
            (
                Abs(position_diff) > old_position * 0.3 AND
                old_position > 0 AND
                (old_position <= 100 OR current_position BETWEEN 1 AND 100)
            ) OR (
                current_position IS NULL AND old_position BETWEEN 1 AND 100
            ) OR (
                old_position IS NULL AND current_position BETWEEN 1 AND 100
            )
        )
    ORDER BY current_position
"""


class TopChangesNotification(ExecuteWithYtTarget):
    title = 'top_changes_notifications'
    dependencies = [TopPartnerChanges]
    final = True

    _from = 'no-reply@yandex-team.ru'
    _fields = ('login', 'target_type', 'current_position', 'old_position', 'position_diff')
    # Адреса из PWISH-58
    _notification_mails = ('partner-managers', 'bonnieclyde', 'yan-partners', 'partner-top-monitor', 'bzzzz')

    _base_template = """
                <html>
                <head>
                    <style>
                        table {
                            font-family: arial, sans-serif;
                            border-collapse: collapse;
                        }

                        td, th {
                            border: 1px solid #dddddd;
                            text-align: left;
                            padding: 8px;
                        }

                        tr:nth-child(even) {
                            background-color: #dddddd;
                        }
                    </style>
                </head>
                <body>
                    <p>Добрый день!<br>%s</p>
                    %s
                </body>
                </html>
            """
    _empty_message = """{date} по сравнению с {week_ago} изменений в топе не было"""
    _data_message = """
Отчеты по изменениям в топе партнеров {date} по сравнению с {week_ago}, сгруппированные по типу устройств:
"""

    @classmethod
    def execute(cls, dependencies_path, yt_client, context):
        assert len(dependencies_path) == 1
        data = yt_client.read_table(dependencies_path[0],
                                    format=yt_client.JsonFormat(attributes={"encode_utf8": False}))

        tables = OrderedDict()

        tables['desktop'] = []
        tables['mobile'] = []
        tables['inapp'] = []

        for item in data:
            tables[item['device_type']].append(item)

        attachments = []
        for device_type, content in tables.items():
            if not content:
                continue

            attachments.append(_generate_pretty_table(device_type, context, content, cls._fields))

        recipients = [(x + '@yandex-team.ru') for x in cls._notification_mails]

        msg = MIMEMultipart()
        msg["Subject"] = '[{}] Изменения в топе партнеров'.format(context['date'])
        msg["To"] = COMMASPACE.join(recipients)
        msg["From"] = cls._from
        msg['Date'] = formatdate(localtime=True)

        if attachments:
            template = cls._data_message
        else:
            template = cls._empty_message

        content = cls._base_template % (template.format(date=context['date'],
                                                        week_ago=context['date'] - timedelta(days=7)),
                                        '<br>'.join(attachments))
        msg.attach(MIMEText(content, 'html', 'utf-8'))

        smtp = SMTP(host='yabacks.yandex.ru', port=25)
        smtp.sendmail(cls._from, recipients, msg.as_string())
        smtp.quit()
        return []


def _generate_pretty_table(device_type, context, data, fields):
    head = _row(_wrap(x, 'th') for x in fields)
    body = ''
    for line in data:
        line['login'] = _login_link(device_type, context, line)
        body += _row(_wrap(line[x]) for x in fields)
    return '<h4>Тип {}</h4><table><thead>{}</thead><tbody>{}</tbody></table>'.format(device_type, head, body)


def _row(line):
    return '<tr>{}</tr>'.format(''.join(line))


def _wrap(item, tag='td'):
    return '<{tag}>{item}</{tag}>'.format(item=item, tag=tag)


def _login_link(device_type, context, line):
    return (
        '<a href="https://api.stat.yandex-team.ru/Direct/Others/yantoppartners?'
        'scale=d&_p2p_amount=1&_p2p_scale=y&device_type={device_type}&key={login}&target_type={target_type}&'
        '_incl_fields=traffic&_incl_fields=traffic__p2p_y_0&_incl_fields=position_min&'
        '_incl_fields=position_min__p2p_y_0&_incl_fields=position_avg&_incl_fields=position_avg__p2p_y_0&'
        '_incl_fields=position_max&_incl_fields=position_max__p2p_y_0&_incl_fields=all_hits&'
        '_incl_fields=all_hits__p2p_y_0&_incl_fields=ad_hits&_incl_fields=ad_hits__p2p_y_0&'
        '_incl_fields=block_shows&_incl_fields=block_shows__p2p_y_0&_incl_fields=direct_block_shows&'
        '_incl_fields=direct_block_shows__p2p_y_0&_incl_fields=shows&_incl_fields=shows__p2p_y_0&'
        '_incl_fields=clicks&_incl_fields=clicks__p2p_y_0&_incl_fields=cost_total&'
        '_incl_fields=cost_total__p2p_y_0&_incl_fields=cost_partner&_incl_fields=cost_partner__p2p_y_0&'
        '_incl_fields=coef&_incl_fields=coef__p2p_y_0&_incl_fields=ctr&_incl_fields=ctr__p2p_y_0&'
        '_incl_fields=cpmv&_incl_fields=cpmv__p2p_y_0&_incl_fields=cpc&_incl_fields=cpc__p2p_y_0&'
        '_incl_fields=fill_rate&_incl_fields=fill_rate__p2p_y_0&date_min={start_date:%Y-%m-%d}+00:00:00&'
        'date_max={end_date:%Y-%m-%d}+23:59:59">{login}</a>'
    ).format(
         device_type=device_type, login=line['login'], target_type=line['target_type'],
         start_date=context['date'] - timedelta(days=7), end_date=context['date']
    )
