# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import logging
import os
import uuid
from datetime import datetime

import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.direct_internal_analytics.laborer_base.context import get_context, serialize_context
from sandbox.projects.direct_internal_analytics.laborer_base.imports import get_target_by_name, add_package_to_path
from sandbox.projects.direct_internal_analytics.laborer_base.processing import process_target_new, get_data_path, \
    TargetProcessorFactory, TargetProcessor
from ..resources import AdAnalyticsRunnerSyncResource, TargetProcessorContentPackage

logger = logging.getLogger(__name__)

TARGETS_PATH = os.path.join(os.path.dirname(os.path.dirname(os.path.dirname(os.path.realpath(__file__)))), 'targets')
PROD_NAMESPACE = 'Production'
PRESTABLE_NAMESPACE = 'Prestable'
USER_NAMESPACE = 'User'
RANDOM_NAMESPACE = 'Random'
EXACT_NAMESPACE = 'Exact'


def _create_worker(parent_task, data_path, target_name, params, dependencies, content_res, context, clean_up=False):
    from projects.direct_internal_analytics.tasks.worker import AdAnalyticsRunnerWorker

    logger.info("Launching %s worker task", target_name)
    subtask = AdAnalyticsRunnerWorker(
        parent_task,
        description="Worker task for target {}".format(target_name),
        dependencies=dependencies,
        content_res=content_res,
        context=serialize_context(context),
        force=params.force,
        yql_token_secret_id = params.yql_token_secret_id,
        yt_token_secret_id = params.yt_token_secret_id,
        appsflyer_api_token_secret_id = params.appsflyer_api_token_secret_id,
        yt_destination_folder=params.yt_destination_folder,
        yt_cluster=params.yt_cluster,
        ch_host=params.ch_host,
        ch_user=params.ch_user,
        ch_pwd_secret_id = params.ch_pwd_secret_id,
        tvm_id=params.tvm_id,
        tvm_secret_yav_secret=params.tvm_secret_yav_secret,
        target_name=target_name,
        clean_up=clean_up,
        kill_timeout=params.kill_timeout,
    )
    subtask.save()

    logger.info("Creating %s sync resource", target_name)
    resource_path = "{}_{}_{}".format(target_name, context['token'], context['date'].isoformat())
    if clean_up:
        resource_path += '_clean_up'
    lock = AdAnalyticsRunnerSyncResource(
        parent_task, "Worker sync resource for target {}".format(target_name), resource_path,
        data_path=data_path, real_task_id=subtask.id,
    )
    logger.info("%s sync resource id is %s", target_name, lock.id)

    subtask.Parameters.result_flag_res = lock
    subtask.save()

    return lock, subtask


class SandboxTargetProcessorFactory(TargetProcessorFactory):
    def __init__(self, parent_task, context, params, subtasks_repository, content_res):
        self._parent_task = parent_task
        self._context = context
        self._params = params
        self._subtasks_repository = subtasks_repository
        self._content_res = content_res

    def get_processor_for(self, target):
        return SandboxTargetProcessor(self._parent_task, target, self._context, self._params, self._subtasks_repository,
                                      self._content_res)

    def get_sandbox_tasks(self):
        return self._subtasks_repository


class SandboxTargetProcessor(TargetProcessor):
    def __init__(self, parent_task, target, context, params, subtasks_repository, content_res):
        self._parent_task = parent_task
        self._target = target
        self._context = context
        self._params = params
        self._subtasks_repository = subtasks_repository
        self._content_res = content_res

        self._target_name = "{}.{}".format(self._target.__module__, self._target.__name__)
        self._data_path = get_data_path(self._target, self._context)

        self._lock = self._get_old_lock()

    def _get_old_lock(self):
        resource = None
        for state in (ctr.State.READY, ctr.State.NOT_READY):
            resource = AdAnalyticsRunnerSyncResource.find(
                state=state,
                attrs={'data_path': self._data_path}
            ).first()
            if resource is not None:
                break
        return resource

    def get_lock(self):
        return self._lock

    def clear_locks(self):
        # TODO(bzzzz): по-настоящему удалять ресурс
        self._lock = None

    def run_processing(self, dependencies_locks):
        self._lock, subtask = _create_worker(self._parent_task, self._data_path, self._target_name, self._params,
                                             dependencies_locks, self._content_res, self._context)
        self._subtasks_repository.append(subtask.enqueue())

    def is_already_processed(self):
        return self._lock is not None

    def clear_data(self):
        pass


class AdAnalyticsRunnerMaster(sdk2.Task):
    """Мастер таска для построения отчетов. Получает название цели, которую нужно построить и собирает все ее
    зависимости, а затем и саму цель.
    Если зависимости уже построены, они по-умолчанию не перестраиваются.
    """

    class Parameters(sdk2.Task.Parameters):
        target = sdk2.parameters.String("Target name", required=True)
        project = sdk2.parameters.String("Project", required=True)
        only_ready_deps = sdk2.parameters.Bool("Build only on existing dependencies. Fail if they do not exist",
                                               default=False)
        force = sdk2.parameters.Bool("Force recalculate everything", default=False)

        with sdk2.parameters.Group("Connection settings") as conn_setting:
            yt_cluster = sdk2.parameters.String("Yt Cluster", default="hahn", required=True)
            yt_destination_folder = sdk2.parameters.String("YT destination folder for result table", default="tmp",
                                                           required=True)
            yql_token_secret_id = sdk2.parameters.YavSecret(
                label="yql token secret id",
                required=True,
                description='secret should contain keys: yt_token',
            )
            yt_token_secret_id = sdk2.parameters.YavSecret(
                label="yt token secret id",
                required=True,
                description='secret should contain keys: yt_token',
            )
            appsflyer_api_token_secret_id = sdk2.parameters.YavSecret(
                label="appsflyer api token secret id",
                required=False,
                description='secret should contain keys: appsflyer_api_token',
            )
            ch_host = sdk2.parameters.String("ClickHouse Host", required=True)
            ch_user = sdk2.parameters.String("ClickHouse User", required=True)
            ch_pwd_secret_id = sdk2.parameters.YavSecret(
                label="clickhouse password secret id",
                required=False,
                description='secret should contain keys: clickhouse_pwd',
            )
            tvm_id = sdk2.parameters.String("TVM app ID", required=False)
            tvm_secret_yav_secret = sdk2.parameters.YavSecret("YaV secret with TVM app secret", required=False)

        with sdk2.parameters.Group("Extended calculation settings") as calc_settings:
            with sdk2.parameters.String("Namespace", required=True, multiline=True) as namespace:
                namespace.values[USER_NAMESPACE] = namespace.Value(default=True)
                namespace.values[PROD_NAMESPACE] = None
                namespace.values[PRESTABLE_NAMESPACE] = None
                namespace.values[RANDOM_NAMESPACE] = None
                namespace.values[EXACT_NAMESPACE] = None
            namespace_exact = sdk2.parameters.String("Exact namespace")
            date = sdk2.parameters.String("Calculation date in ISO format")
            date_shift = sdk2.parameters.String("Default date shift")
            date_step = sdk2.parameters.String("Step")
            expiration_days = sdk2.parameters.Integer("Results expiration days (0=infinite)")
            date_auto_refill_shift = sdk2.parameters.String("Date autorefill shift")
            custom_params = sdk2.parameters.Dict("Custom context params for target templates")
            content_res = sdk2.parameters.Resource(
                "Targets content resource",
                resource_type=TargetProcessorContentPackage
            )

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logger.info("Starting execution")
        subtasks = list(self.find().limit(0))
        if not subtasks:
            self.run_subtasks(subtasks)

        self.check_subtasks(subtasks)

    def get_resource(self):
        prestable = self.Parameters.namespace == PRESTABLE_NAMESPACE

        if self.Parameters.content_res:
            content_res = self.Parameters.content_res
        elif self.Context.content_res_id == ctm.NotExists:
            attrs = {'resource_name': self.Parameters.project}
            attrs['released'] = 'stable'
            content_res = TargetProcessorContentPackage.find(state=ctr.State.READY, attrs=attrs).order(-TargetProcessorContentPackage.id).first()
            if prestable:
                attrs['released'] = 'prestable'
                content_res_prestable = TargetProcessorContentPackage.find(state=ctr.State.READY, attrs=attrs).order(-TargetProcessorContentPackage.id).first()
                if (content_res is  None) or (content_res_prestable is not None and content_res_prestable.id > content_res.id):
                    content_res = content_res_prestable
        else:
            content_res = TargetProcessorContentPackage.find(id=self.Context.content_res_id).first()

        if content_res is None:
            logger.info('No matching resource found')
            # TODO(bzzzz): Когда полностью перейдем на контент из ресурсов, нужно будет возбуждать исключение
            # raise RuntimeError('No matching resource found')

        return content_res

    def get_namespace(self, resource):
        if self.Context.namespace != ctm.NotExists:
            return self.Context.namespace

        if self.Parameters.namespace == PROD_NAMESPACE and not self.Parameters.content_res:
            namespace = 'production'
        elif self.Parameters.namespace == PRESTABLE_NAMESPACE and resource is not None:
            namespace = 'prestable'
        elif self.Parameters.namespace == RANDOM_NAMESPACE:
            namespace = str(uuid.uuid4())[:10]
        elif self.Parameters.namespace == USER_NAMESPACE:
            namespace = self.author
        elif self.Parameters.namespace_exact:
            namespace = self.Parameters.namespace_exact
        else:
            raise RuntimeError('Namespace not set')

        self.Context.namespace = namespace
        return namespace

    def get_target(self, resource):
        if resource is not None:
            add_package_to_path(str(sdk2.ResourceData(resource).path))
        return get_target_by_name(self.Parameters.target)

    def format_date(self, date):

        date = date + ":0"
        return int(date.split(':')[0]) * 24 + int(date.split(':')[1])

    def get_context(self, namespace, target):
        date = None
        if self.Parameters.date:
            date = datetime.strptime(self.Parameters.date, "%Y-%m-%d %H:%M:%S")

        date_shift = 24
        if self.Parameters.date_shift:
            date_shift = self.format_date(self.Parameters.date_shift)

        date_auto_refill_shift = date_shift
        if self.Parameters.date_auto_refill_shift:
            date_auto_refill_shift = self.format_date(self.Parameters.date_auto_refill_shift)
            if date_auto_refill_shift < date_shift:
                date_auto_refill_shift = date_shift

        date_step = 24
        if self.Parameters.date_step:
            date_step = self.format_date(self.Parameters.date_step)

        expiration_days = 0
        if self.Parameters.expiration_days:
            expiration_days = int(self.Parameters.expiration_days)

        context = get_context(
            date=date,
            token=namespace,
            params=self.Parameters.custom_params or {},
            date_shift=date_shift,
            date_step=date_step,
            date_auto_refill_shift=date_auto_refill_shift,
            target=target,
            expiration_days=expiration_days,
            yt_cluster=self.Parameters.yt_cluster,
            home=self.Parameters.yt_destination_folder
        )
        context['home'] = self.Parameters.yt_destination_folder

        logger.info("Preparing workers with context %s", context)
        return context

    def get_context_and_target(self):
        content = self.get_resource()
        target = self.get_target(content)

        namespace = self.get_namespace(content)
        context = self.get_context(namespace, target)

        return context, target, content

    def run_subtasks(self, subtasks):
        context, target, content_res = self.get_context_and_target()

        factory = SandboxTargetProcessorFactory(self, context, self.Parameters, subtasks, content_res)
        process_target_new(target, factory, with_dependencies=not self.Parameters.only_ready_deps,
                           force=self.Parameters.force)

    def check_subtasks(self, subtasks):
        """Проверить из залогировать состояние подзадач и ресурсов синхронизации.
        Если одна из подзадач непоправимо зафейлилась, остановить все подзадачи и мастер-таску
        """
        logger.info("Checking child tasks of master task")

        running = []
        waiting = []
        stop_subtasks = False
        for subtask in subtasks:
            logger.info("Subtask #%s status: %s", subtask.id, subtask.status)
            if subtask.status in ctt.Status.Group.BREAK or subtask.status == ctt.Status.FAILURE:
                stop_subtasks = True
            elif subtask.status in ctt.Status.Group.WAIT:
                waiting.append(subtask)
            elif subtask.status not in ctt.Status.Group.FINISH and subtask.status != ctt.Status.FINISHING:
                running.append(subtask)

        if waiting or running:
            subtasks_to_wait = running[:]
            if stop_subtasks:
                logger.info("Stopping waiting subtasks because exception occured")
                for subtask in waiting:
                    try:
                        subtask.stop()
                    except TaskError:
                        pass
            else:
                subtasks_to_wait.extend(waiting)

            if subtasks_to_wait:
                logger.info("Waiting for %s tasks to finish", len(subtasks_to_wait))
                subtasks_to_wait = [s.id for s in subtasks_to_wait]
                raise sdk2.WaitTask(subtasks_to_wait, list(ctt.Status.Group.FINISH.union(ctt.Status.Group.BREAK)),
                                    wait_all=False)

        if stop_subtasks:
            logger.info("Stopping master task after an error in child task")
            self.stop()
        elif self.Context.clean_up == ctm.NotExists:
            self.Context.clean_up = True
            self.launch_cleaner()

    def launch_cleaner(self):
        logger.info("Launching cleaner worker")
        context, target, content_res = self.get_context_and_target()
        context['date'] = context['clean_date']

        _, subtask = _create_worker(self, get_data_path(target, context) + "_cleanup",
                                    "{}.{}".format(target.__module__, target.__name__),
                                    self.Parameters, [], content_res, context, clean_up=True)
        subtask.enqueue()
        raise sdk2.WaitTask([subtask.id], list(ctt.Status.Group.FINISH.union(ctt.Status.Group.BREAK)),
                            wait_all=True)
