# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import logging
from datetime import timedelta, datetime

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from ..master import AdAnalyticsRunnerMaster

logger = logging.getLogger(__name__)


class AdAnalyticsRunnerScheduler(AdAnalyticsRunnerMaster):
    """Таска для последовательного построения отчетов за несколько дат"""

    class Parameters(AdAnalyticsRunnerMaster.Parameters):
        with sdk2.parameters.Group("Period settings") as period_setting:
            start_date = sdk2.parameters.String("Calculation start date in ISO format", required=True)
            end_date = sdk2.parameters.String("Calculation end date in ISO format", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logger.info("Starting execution")
        start_date = datetime.strptime(self.Parameters.start_date, "%Y-%m-%d").date()

        if self.Context.current_date == ctm.NotExists:
            current_date = datetime.strptime(self.Parameters.end_date, "%Y-%m-%d").date()
        else:
            current_date = datetime.strptime(self.Context.current_date, "%Y-%m-%d").date() - timedelta(days=1)

        if current_date < start_date:
            return

        self.Context.current_date = current_date.isoformat()
        logger.info("Current date is %s", current_date)

        self.run_subtask(current_date)

    def run_subtask(self, current_date):
        params = dict({k: v for k, v in self.Parameters})
        params.pop("start_date")
        params.pop("end_date")
        params['date'] = current_date.isoformat()

        logger.info("Starting master task with params %s", params)
        subtask = AdAnalyticsRunnerMaster(
            self,
            description="Master for date {}".format(current_date),
            **params
        )
        subtask.save()
        subtask.enqueue()

        raise sdk2.WaitTask([subtask], list(ctt.Status.Group.FINISH.union(ctt.Status.Group.BREAK)), wait_all=False)
