# coding=utf-8
from __future__ import absolute_import, unicode_literals, print_function

import logging
import os
import tempfile
from ConfigParser import SafeConfigParser

import sys

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.projects.direct_internal_analytics.laborer_base import const
from sandbox.projects.direct_internal_analytics.laborer_base.imports import add_package_to_path
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers.process import subprocess as sp
from ..resources import AdAnalyticsRunnerSyncResource, TargetProcessorContentPackage

logger = logging.getLogger(__name__)


class AdAnalyticsRunnerWorker(sdk2.Task):
    """Таска непосредственно строящая отчет по заданной цели.
    Предполагается, что она запускается в тот момент, когда все данные зависимостей уже готовы.
    Сейчас синхронизация зависимостей делается через ожидание готовности ресурсов
    """
    SUBPROCESS_RUNNER_NAME = 'make_target'

    class Parameters(sdk2.Task.Parameters):
        result_flag_res = sdk2.parameters.ParentResource(
            "Processing result",
            resource_type=AdAnalyticsRunnerSyncResource
        )
        dependencies = sdk2.parameters.Resource(
            "Dependencies",
            resource_type=AdAnalyticsRunnerSyncResource,
            multiple=True
        )
        content_res = sdk2.parameters.Resource(
            "Targets content",
            resource_type=TargetProcessorContentPackage
        )
        force = sdk2.parameters.Bool("Force recalculate everything")
        clean_up = sdk2.parameters.Bool("Clean for target", default=False)

        context = sdk2.parameters.String("Calculation context")
        target_name = sdk2.parameters.String("Target name")

        yql_token_secret_id = sdk2.parameters.YavSecret(
            label="yql token secret id",
            required=True,
            description='secret should contain keys: yt_token',
        )
        yt_token_secret_id = sdk2.parameters.YavSecret(
            label="yt token secret id",
            required=True,
            description='secret should contain keys: yt_token',
        )
        appsflyer_api_token_secret_id = sdk2.parameters.YavSecret(
            label="appsflyer api token secret id",
            required=False,
            description='secret should contain keys: appsflyer_api_token',
        )
        yt_destination_folder = sdk2.parameters.String("YT destination folder for result table",
                                                       default="tmp", required=True)
        yt_cluster = sdk2.parameters.String("Yt Cluster", default="hahn", required=True)
        ch_host = sdk2.parameters.String("ClickHouse Host", required=True)
        ch_user = sdk2.parameters.String("ClickHouse User", required=True)
        ch_pwd_secret_id = sdk2.parameters.YavSecret(
            label="clickhouse password secret id",
            required=False,
            description='secret should contain keys: clickhouse_pwd',
        )
        tvm_id = sdk2.parameters.String("TVM app ID", required=False)
        tvm_secret_yav_secret = sdk2.parameters.YavSecret("YaV secret with TVM app secret", required=False)

        debug = sdk2.parameters.Bool("Run task in debug mode", default=False)

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 1
        disk_space = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def _create_config_file(self, fobj):
        cp = SafeConfigParser()
        cp.add_section(const.C_SECTION)
        cp.set(const.C_SECTION, const.C_YQL_TOKEN,
               self.Parameters.yql_token_secret_id.data()['yt_token'])
        cp.set(const.C_SECTION, const.C_YT_TOKEN,
               self.Parameters.yt_token_secret_id.data()['yt_token'])

        if self.Parameters.ch_pwd_secret_id:
            cp.set(const.C_SECTION, const.C_CH_PWD,
                   self.Parameters.ch_pwd_secret_id.data()['clickhouse_pwd'])
        if self.Parameters.tvm_secret_yav_secret:
            cp.set(const.C_SECTION, const.C_TVM_SECRET,
                   self.Parameters.tvm_secret_yav_secret.data()['client_secret'])
        if self.Parameters.appsflyer_api_token_secret_id:
            cp.set(const.C_SECTION, const.C_APPSFLYER_API_TOKEN,
                   self.Parameters.appsflyer_api_token_secret_id.data()['appsflyer_api_token'])

        cp.write(fobj.file)
        fobj.file.flush()

        if self.Parameters.debug:
            with open(fobj.name) as f:
                logging.info("Configuration file contents: %s", f.read())

    def on_execute(self):
        # Выкачиваем нужные таргеты и добавляем их в PYTHONPATH
        unpacked_target_path = None
        if self.Parameters.content_res:
            unpacked_target_path = add_package_to_path(unicode(sdk2.ResourceData(self.Parameters.content_res).path))

        logging.info("Creating virtualenv")
        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('setuptools', version="44.0.0", venv=venv).prepare()
            environments.PipEnvironment('pip', venv=venv, version=environments.PipEnvironment.PIP_VERSION).prepare()
            environments.PipEnvironment('yql', venv=venv, version='1.2.89').prepare()
            environments.PipEnvironment('clickhouse', venv=venv, version='0.1.6').prepare()
            environments.PipEnvironment('yandex-yt-transfer-manager-client', venv=venv, version='0.0.31.post0').prepare()
            environments.PipEnvironment('yandex-yt', venv=venv, version='0.9.26').prepare()
            environments.PipEnvironment('jinja2', venv=venv, version='2.9.6').prepare()

            script_path = os.path.join(os.path.dirname(os.path.dirname(os.path.dirname(os.path.realpath(__file__)))),
                                       'laborer', 'scripts', '{}.py'.format(self.SUBPROCESS_RUNNER_NAME))

            with tempfile.NamedTemporaryFile('w', bufsize=0) as cf:
                self._create_config_file(cf)

                process_description = [
                    venv.executable, script_path,
                    const.YT_CLUSTER, self.Parameters.yt_cluster,
                    const.CH_HOST, self.Parameters.ch_host,
                    const.CH_USER, self.Parameters.ch_user,
                    const.TVM_ID, self.Parameters.tvm_id,
                    const.CONFIG, cf.name,
                    const.DESTINATION, self.Parameters.yt_destination_folder,
                    const.CONTEXT, self.Parameters.context,
                    const.TARGET, self.Parameters.target_name,
                ]
                if unpacked_target_path is not None:
                    process_description.extend([const.UNPACKED_TARGET_PATH, unpacked_target_path])

                if self.Parameters.clean_up:
                    process_description.append(const.CLEAN_UP_ONLY)
                elif self.Parameters.force:
                    process_description.append(const.FORCE)
                env = {'PYTHONPATH': ':'.join(sys.path)}

                logging.info("Launching process %s with env %s", process_description, env)

                with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
                    p = sp.Popen(process_description, stdout=pl.stdout, stderr=pl.stderr, env=env)
                    code = p.wait()

            if code != 0:
                raise RuntimeError("Process exited with code {}. Check common log".format(code))

        logging.info("Successfully finished process execution")
        if self.Parameters.result_flag_res:
            msg = "Done in task {}".format(self.id)

            # Магия для того, чтобы задача не падала на выходе из за бага/странной фичи в сендбоксе
            common.fs.allocate_file(unicode(self.Parameters.result_flag_res.path), len(msg.encode('utf-8')))
