# coding: utf-8

import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.sdk2.helpers import subprocess as sp


class BuildDiskDockerImage(sdk2.Task):
    """
    Build disk docker image by path and tag
    """
    class Requirements(sdk2.Task.Requirements):
        client_tags = (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL)
        dns = ctm.DnsType.DNS64
        disk_space = 51200
        ram = 4096
        # LXC_CONTAINER_WITH_DOCKER https://sandbox.yandex-team.ru/resource/773239891
        container_resource = 773239891

    class Parameters(sdk2.Task.Parameters):
        docker_image_path = sdk2.parameters.String('Docker image path', required=True)
        revision = sdk2.parameters.String('Revision', required=True)

    def build_command(self):
        command = ['/bin/bash']
        if self.Parameters.docker_image_path.startswith('disk/clusters'):
            image_name = self.Parameters.docker_image_path.split('/')[-1]
            command += ['build_one_cluster.sh', image_name, self.Parameters.revision]
        elif self.Parameters.docker_image_path.startswith('disk/common'):
            command.append('build_common.sh')
        elif self.Parameters.docker_image_path.startswith('disk/distro'):
            command.append('build_distro.sh')
        else:
            return ['/bin/echo', self.Parameters.docker_image_path]
        return command

    def on_execute(self):
        os.environ['SANDBOX_TOKEN'] = sdk2.Vault.data(self.owner, 'SANDBOX_TOKEN')
        os.environ['DOCKER_AUTH'] = sdk2.Vault.data(self.owner, 'DOCKER_AUTH')
        os.environ['TESTENV_TOKEN'] = sdk2.Vault.data(self.owner, 'TESTENV_TOKEN')
        os.environ['CURRENT_REVISION'] = self.Parameters.revision
        arcadia_url = "arcadia:/arc/trunk/arcadia/disk/admin/docker/@%s" % self.Parameters.revision
        command = self.build_command()
        with arcadiasdk.mount_arc_path(arcadia_url) as checkout_path:
            with sdk2.helpers.ProcessLog(self, logger=self.Parameters.docker_image_path.split('/')[-1]) as pl:
                for attempt in range(3):
                    pl.logger.info('Attempt %d, run docker build', attempt)
                    try:
                        sp.check_call(command, stdout=pl.stdout, stderr=pl.stderr, cwd=checkout_path)
                    except sp.CalledProcessError:
                        pl.logger.warning('Attempt %d failed', attempt)
                    else:
                        break
                else:
                    raise TaskFailure('Failed to build image: %s' % self.Parameters.docker_image_path)
