# coding: utf-8

import ast
import os

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.disk.BuildDiskDockerImage import BuildDiskDockerImage


class BuildDiskDockerImages(sdk2.Task):
    """
    Build all images by paths
    """
    class Parameters(sdk2.Task.Parameters):
        commit_paths = sdk2.parameters.String('Commit paths', required=True)
        revision = sdk2.parameters.String('Revision', required=True)

    class Context(sdk2.Task.Context):
        child_tasks_ids = []

    def get_all_clusters(self):
        arcadia_url = "arcadia:/arc/trunk/arcadia/disk/admin/docker/disk/clusters/@%s" % self.Parameters.revision
        with arcadiasdk.mount_arc_path(arcadia_url) as checkout_path:
            return [p for p in os.listdir(checkout_path) if p != '__SVNVERSION__']

    def on_execute(self):
        prefix = '/trunk/arcadia/disk/admin/docker/disk/clusters/'
        if not self.Context.child_tasks_ids:
            clusters = set(
                x[len(prefix):].split('/')[0]
                for x in ast.literal_eval(self.Parameters.commit_paths)
                if x.startswith(prefix)
            )
            for cluster in sorted(clusters or self.get_all_clusters()):
                child = BuildDiskDockerImage(
                    self,
                    docker_image_path='disk/clusters/%s' % cluster,
                    revision=self.Parameters.revision,
                    owner=self.owner,
                    description="Build disk/clusters/%s" % cluster,
                )
                child.enqueue()
                self.Context.child_tasks_ids.append(child.id)

        if not all(sdk2.Task[t].status in ctt.Status.Group.FINISH for t in self.Context.child_tasks_ids):
            raise sdk2.WaitTask(self.Context.child_tasks_ids, ctt.Status.Group.FINISH, wait_all=True)

        success_tasks = [sdk2.Task[t].status == ctt.Status.SUCCESS for t in self.Context.child_tasks_ids]
        if not all(success_tasks):
            raise TaskFailure('Failed %s images' % len(filter(lambda x: not x, success_tasks)))
        self.set_info('Built %s images' % len(success_tasks))
