# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.types import task
from sandbox.projects.common import file_utils
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

import logging
import logging.handlers
import os
import time


ARCADIA_CONFIG_DIR = 'devtools/distbuild/lunapark'
TANKS = ['hellcat.tanks.yandex.net', 'lee.tanks.yandex.net', 'scorpion.tanks.yandex.net']  # some SAS tanks
LOGS_TO_SAVE = ['answ*.log', 'tank*.log', 'jobno*.txt', 'pandora*.log']


def logger():
    logger = logging.getLogger('%s_%s' % (__name__, time.time()))
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'distbuild_shooting.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    logger.addHandler(file_handler)
    return logger


class DistbuildLoadTesting(sdk2.Task):
    """Distbuild Load Testing"""

    class Requirements(sdk2.Requirements):
        semaphores = task.Semaphores(
            acquires=[
                task.Semaphores.Acquire(
                    name='distbuild_load_shooting',
                    weight=1,
                )
            ],
            release=(task.Status.Group.BREAK, task.Status.Group.FINISH)
        )

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Output:
            lunapark_job_ids = sdk2.parameters.List('List of Lunapark jobs', default=[], description='List of Lunapark jobs')

    class Context(sdk2.Context):
        shooting_task_ids = []

    def start_shooting(self, desc, config_content, tanks):
        subtask_shoot = ShootViaTankapi(
            self,
            description=desc,
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            logs_to_save=LOGS_TO_SAVE,
        ).enqueue()
        self.Context.shooting_task_ids.append(subtask_shoot.id)
        logger().info('Subtask with shooting is started')
        raise sdk2.WaitTask([subtask_shoot.id], task.Status.Group.FINISH | task.Status.Group.BREAK, wait_all=True, timeout=14400)

    def on_prepare(self):
        os.makedirs('configs', 0o755)
        self.local_configs = os.path.join(str(self.path('configs')), 'shoots')
        tank_config_dir = sdk2.svn.Arcadia.trunk_url(os.path.join(ARCADIA_CONFIG_DIR, 'configs'))
        sdk2.svn.Arcadia.export(tank_config_dir, self.local_configs)

    def on_execute(self):
        tank_configs = []
        for root, _, confs in os.walk(self.local_configs):
            for conf in confs:
                if conf.endswith('.yaml'):
                    tank_configs.append(os.path.join(root, conf))
        tank_configs.sort()
        logger().info('Tank configs: %s', tank_configs)

        shooting_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        logger().info('Shooting dir: %s', shooting_dir)

        for conf_name in tank_configs:
            with self.memoize_stage["shooting_{0}".format(conf_name)]:
                config_content = file_utils.read_file(conf_name)
                logger().info('Config: %r', config_content)
                self.start_shooting(conf_name, config_content, TANKS)

        # check after all shootings finished
        exception_messages = []
        fail_messages = []
        lunapark_job_ids = []
        for subtask_id in self.Context.shooting_task_ids:
            subtask = sdk2.Task.find(id=subtask_id).first()
            if subtask.status in (task.Status.EXCEPTION, task.Status.TIMEOUT):
                exception_messages += ['Exception: shooting task {} finished with {}'.format(subtask_id, subtask.status)]
            elif subtask.status != task.Status.SUCCESS:
                exception_messages += ['Fail: shooting task {} finished with {}'.format(subtask_id, subtask.status)]
            lunapark_job_ids.append(subtask.Parameters.lunapark_job_id)
        self.Parameters.lunapark_job_ids = lunapark_job_ids  # NOTE: parameters are set only on direct assignment

        if exception_messages:
            raise TaskError('\n'.join(exception_messages))
        elif fail_messages:
            raise TaskFailure('\n'.join(fail_messages))
