from sandbox import sdk2

from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox.common import errors
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard

TEST = 'testing'
PRIEMKA = 'prestable'
PROD = 'stable'

TASK_RELEASE_TYPE = {
    TEST: ctt.ReleaseStatus.TESTING,
    PRIEMKA: ctt.ReleaseStatus.PRESTABLE,
    PROD: ctt.ReleaseStatus.STABLE
}


class DjShardAutoDeploy(sdk2.Task):
    """ Task for shard release and deployment """

    class Parameters(sdk2.Task.Parameters):
        shardmap_task_id = sdk2.parameters.Integer("Id of publish shards task", required=True)
        shard_env = sdk2.parameters.String(
            "Shard environment",
            choices=[('Production', PROD), ('Prestable', PRIEMKA), ('Test', TEST)],
            required=True,
        )
        deploy_shard = sdk2.parameters.Bool("Deploy shard", required=True)
        resource_name = sdk2.parameters.String("Sandbox resource name", required=True)

        nanny_dashboard_name = sdk2.parameters.String("Nanny dashboard name", required=True)
        nanny_dashboard_recipe = sdk2.parameters.String("Nanny dashboard recipe", required=True)
        nanny_dashboard_filter = sdk2.parameters.String("Nanny dashboard filter", required=True)

        vault_name = sdk2.parameters.String("Vault name", default='nanny_oauth_token', required=True)
        vault_owner = sdk2.parameters.String("Vault owner", required=True)
        semaphore_name = sdk2.parameters.String("Semaphore Name", required=True)

        release_title = sdk2.parameters.String("Release title", default="Autodeploy", required=False)
        services_from_recipe = sdk2.parameters.Bool("Get list of services from recipe", default=False, required=False)

    def find_shard_task_and_resource(self):
        shard_task = sdk2.Task.find(id=self.Parameters.shardmap_task_id).first()
        if shard_task is None:
            raise common.errors.TaskFailure("Can't find shard task")

        self.Context.shard_task_id = shard_task.id

        shard_resource = sdk2.Resource[self.Parameters.resource_name].find(task=shard_task).first()

        if shard_resource is None:
            raise common.errors.TaskFailure("Can't find shard resource")
        self.Context.shard_resource_id = shard_resource.id

        self.Context.save()

    def release_shard_task(self):
        self.server.release(
            task_id=self.Context.shard_task_id,
            type=TASK_RELEASE_TYPE[self.Parameters.shard_env],
            subject=self.Parameters.release_title
        )

    def run_dashboard_task(self):
        deploy_task_params = {
            'deployment_nanny_dashboard_name': self.Parameters.nanny_dashboard_name,
            'deployment_nanny_dashboard_recipe': self.Parameters.nanny_dashboard_recipe,
            'deployment_nanny_dashboard_filter': self.Parameters.nanny_dashboard_filter,
            'deployment_task_id': self.Context.shard_task_id,
            'deployment_release_status': self.Parameters.shard_env,
            'vault_name': self.Parameters.vault_name,
            'vault_owner': self.Parameters.vault_owner,
            'deployment_nanny_bool_wait': True,
            'semaphore_name': self.Parameters.semaphore_name,
            'services_from_recipe': self.Parameters.services_from_recipe,
        }
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def check_deploy_task(self):
        if self.Context.deploy_task_id is not NotExists:
            child = sdk2.Task[self.Context.deploy_task_id]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Deploy task was finished with status {}".format(child.status))

    def on_execute(self):
        with self.memoize_stage.find_shard:
            self.find_shard_task_and_resource()

        with self.memoize_stage.final_deploy:
            self.release_shard_task()
            if self.Parameters.deploy_shard:
                self.run_dashboard_task()

        with self.memoize_stage.check_deploy:
            self.check_deploy_task()
