# -*- coding: utf-8 -*-

import os

from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common.build.KosherYaMake import KosherYaMake
from sandbox.projects.common.nanny import nanny

from sandbox.projects.dj.resource_types import (
    DjViewerExecutable,
    DjViewerFront,
    DjViewerConfig,
)


class BuildDjViewer(KosherYaMake):
    """Build DJ viewer"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL

    class Parameters(KosherYaMake.Parameters):
        viewer_executable_resource_type = sdk2.parameters.String('Viewer executable resource type', default_value=DjViewerExecutable.name)
        viewer_front_resource_type = sdk2.parameters.String('Viewer front resource type', default_value=DjViewerFront.name)
        viewer_config_resource_type = sdk2.parameters.String('Viewer config resource type', default_value=DjViewerConfig.name)

        with sdk2.parameters.String('Release to Nanny') as release_to_nanny:
            release_to_nanny.values[''] = '-'
            for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.STABLE]:
                release_to_nanny.values[status] = status

        ext_params = binary_task.binary_release_parameters(none=True)

    @staticmethod
    def _get_resources_if_defined(resource_type_names):
        resource_types = []
        for resource_type_name in resource_type_names:
            if resource_type_name:
                resource_types.append(sdk2.Resource[resource_type_name])
        return resource_types

    def _get_build_resources(self):
        return self._get_resources_if_defined([self.Parameters.viewer_executable_resource_type, self.Parameters.viewer_front_resource_type])

    def _get_source_resources(self):
        return self._get_resources_if_defined([self.Parameters.viewer_config_resource_type])

    def get_resources(self):
        resources = {}
        for resource_type in self._get_build_resources() + self._get_source_resources():
            resources[resource_type.name] = {
                'resource_type': resource_type,
                'resource_path': os.path.basename(resource_type.arcadia_build_path),
                'description': resource_type.name,
            }
        return resources

    def get_targets(self):
        return [resource_type.arcadia_build_path for resource_type in self._get_build_resources()]

    def get_arts(self):
        return [{'path': resource_type.arcadia_build_path} for resource_type in self._get_build_resources()]

    def get_arts_source(self):
        return [{'path': resource_type.arcadia_build_path} for resource_type in self._get_source_resources()]

    def on_success(self, prev_status):
        if self.Parameters.release_to_nanny:
            additional_parameters = dict(
                releaser=self.author,
                release_status=self.Parameters.release_to_nanny,
                release_subject='Viewer automatic release',
                email_notifications=dict(to=[], cc=[]),
            )
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
