# -*- coding: utf-8 -*-

from sandbox.projects import resource_types

from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import process

import os
import logging
from sandbox import sdk2


'''
Build recommender index shard with given number.
'''


class BuildRecommenderShard(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='banach')
        yt_token = sdk2.parameters.String('YT token vault', required=True)
        shard_number = sdk2.parameters.Integer('Shard number', required=True)
        shard_name = sdk2.parameters.String('Shard name', required=True)
        index_state_dir = sdk2.parameters.String('Index state dir', required=True)
        shard_resource = sdk2.parameters.String('Shard resource type', required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet", version="0.3.32-0"),  # DEVTOOLSSUPPORT-5654
        )
        ram = 64 * 1024
        disk_space = 100 * 1024

    def download_binaries(self):
        import yt.wrapper as yt

        exec_path = str(self.path('index_builder'))
        config_path = str(self.path('config.pbtxt'))

        yt.config.set_proxy(self.Parameters.yt_proxy)
        yt.config['token'] = sdk2.Vault.data(self.Parameters.yt_token)

        stream = yt.read_file(yt.ypath_join(self.Parameters.index_state_dir, 'binaries', 'index_builder'))
        with open(exec_path, 'wb') as f:
            f.write(stream.read())

        stream = yt.read_file(yt.ypath_join(self.Parameters.index_state_dir, 'config', 'config.pbtxt'))
        with open(config_path, 'wb') as f:
            f.write(stream.read())

        return exec_path, config_path

    def _get_iss_shards_tool(self):
        try:
            tool_id = utils.get_and_check_last_released_resource_id(
                resource_types.ISS_SHARDS,
                arch=sandboxapi.ARCH_ANY
            )
            return sdk2.ResourceData(sdk2.Resource.find(id=tool_id).first()).path
        except Exception as e:
            logging.error("Cannot get latest stable ISS_SHARDS tool: %s", e)

    def on_execute(self):
        (shard_writer, config_path) = self.download_binaries()
        os.chmod(shard_writer, 0775)

        resource = sdk2.Resource[self.Parameters.shard_resource](self,
            'Recommender index shard - ' + self.Parameters.shard_name, self.Parameters.shard_name)
        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        shard_dir = str(data.path)

        env = os.environ.copy()
        env["YT_TOKEN"] = sdk2.Vault.data(self.Parameters.yt_token)

        cmd = [
            shard_writer,
            'build_shard',
            '--server', self.Parameters.yt_proxy,
            '--state_dir', self.Parameters.index_state_dir,
            '--shard_number', str(self.Parameters.shard_number),
            '--output_dir', shard_dir,
            '--config', config_path
        ]

        process.run_process(cmd, work_dir=shard_dir, environment=env, log_prefix='index_builder')

        iss_shards = str(self._get_iss_shards_tool())
        print (iss_shards)
        process.run_process(
                [
                    iss_shards,
                    'configure',
                    shard_dir,
                    '--id',
                    self.Parameters.shard_name
                ],
                work_dir=str(self.path()),
                log_prefix='iss_shards_configure'
        )

        process.run_process(
                [
                    iss_shards,
                    'register',
                    shard_dir
                ],
                work_dir=str(self.path()),
                log_prefix='iss_shards_register'
        )

        data.ready()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
