import logging
import os

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.apihelpers import get_last_resource_with_attribute
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.dj import DjBuildShard2ResourceBundle


class DjAutoDeployShard2Models(sdk2.Task):
    """ Task for DJ shard2.0 models auto deploy """

    class Parameters(sdk2.Task.Parameters):
        nanny_token_vault = sdk2.parameters.String(
            'Nanny OAuth token vault',
            required=True
        )

        nanny_token_vault_owner = sdk2.parameters.String(
            'Nanny OAuth token vault owner',
            required=True
        )

        nanny_service_name = sdk2.parameters.String(
            'Nanny service name',
            required=True
        )

        nanny_semaphore_name = sdk2.parameters.String(
            'Nanny semaphore name',
            required=False
        )

        models_resource = sdk2.parameters.Resource(
            'Models resource',
            required=True
        )

        bundle_resource_type = sdk2.parameters.String(
            'Bundle resource type',
            default='DJ_TUTORIAL_RESOURCE_BUNDLE',
            required=True
        )

        deployment_nanny_dashboard_name = sdk2.parameters.String(
            'Dashboard name',
            required=True
        )

        deployment_nanny_dashboard_recipe = sdk2.parameters.String(
            'Dashboard recipe',
            required=True
        )

        deployment_nanny_dashboard_filter = sdk2.parameters.String(
            'Dashboard filter',
            required=False
        )

    def create_nanny_client(self):
        nanny_token = sdk2.Vault.data(self.Parameters.nanny_token_vault_owner, self.Parameters.nanny_token_vault)
        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=nanny_token,
        )

    def find_production_bundle_task(self):
        logging.info("Creating nanny client...")
        nanny_client = self.create_nanny_client()
        current_resources = nanny_client.get_service_resources(self.Parameters.nanny_service_name)
        for files in current_resources['content']['sandbox_files']:
            if files['resource_type'] == self.Parameters.bundle_resource_type:
                return sdk2.Task[files['task_id']]
        logging.warning("Can not find bundle resource")
        return None

    def prepare_bundle_task(self):
        logging.info("Search for current production bundle...")
        current_production_bundle_task = self.find_production_bundle_task()
        logging.info("Current production bundle task id {}".format(current_production_bundle_task.id))

        logging.info("Prepare new bundle params...")
        params = dict(current_production_bundle_task.Parameters)
        params['models_resource'] = self.Parameters.models_resource
        return sdk2.Task[current_production_bundle_task.type.name](self, **params)

    def run_bundle_task(self):
        logging.info("Start bundle building...")
        new_bundle_task = self.prepare_bundle_task()
        new_bundle_task.save().enqueue()
        self.Context.new_bundle_task_id = new_bundle_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.new_bundle_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def find_new_bundle_resource(self):
        logging.info("Search for new bundle resource...")
        resource = sdk2.Resource.find(
            self.bundle_resource_type,
            task=sdk2.Task[self.Context.new_bundle_task_id]
        ).first()
        logging.info("New bundle resource id : {}".format(resource.id))
        return resource

    def release_bundle_task(self):
        logging.info("Release bundle task...")
        self.server.release(
            task_id=self.Context.new_bundle_task_id,
            type=ctt.ReleaseStatus.STABLE,
            subject="Models autodeploy from sandbox task {}".format(self.id)
        )

    def run_dashboard_task(self):
        logging.info("Run dashboard task...")
        deploy_task_params = {
            'deployment_nanny_dashboard_name': self.Parameters.deployment_nanny_dashboard_name,
            'deployment_nanny_dashboard_recipe': self.Parameters.deployment_nanny_dashboard_recipe,
            'deployment_nanny_dashboard_filter': self.Parameters.deployment_nanny_dashboard_filter,
            'deployment_task_id': self.Context.new_bundle_task_id,
            'deployment_release_status': 'stable',
            'vault_name': self.Parameters.nanny_token_vault,
            'vault_owner': self.Parameters.nanny_token_vault_owner,
            'deployment_nanny_bool_wait': True,
            'semaphore_name': self.Parameters.nanny_semaphore_name
        }
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run production deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def deploy_bundle(self):
        self.release_bundle_task()
        self.run_dashboard_task()

    def check_deploy(self):
        logging.info("Check deploy task status...")
        child = sdk2.Task[self.Context.deploy_task_id]
        if child.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Deploy task was finished with status {}".format(child.status))

    def release_models(self):
        logging.info("Release models task...")
        self.server.release(
            task_id=self.Parameters.models_resource.task_id,
            type=ctt.ReleaseStatus.STABLE,
            subject="Autodeploy from sandbox task {}".format(self.id)
        )

    def on_execute(self):
        with self.memoize_stage.build_bundle:
            self.run_bundle_task()

        with self.memoize_stage.deploy_bundle:
            self.deploy_bundle()

        with self.memoize_stage.release_models:
            self.check_deploy()
            self.release_models()
