import logging
import os
import tarfile

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
import sandbox.common.types.task as ctt


class DjBuildRecommenderBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Task for building recommender bundle """

    class Parameters(sdk2.Task.Parameters):
        bundle_resource_name = sdk2.parameters.String(
            "Sandbox resource name for bundle",
            required=True
        )

        svn_revision = sdk2.parameters.Integer("SVN revision", required=False)

        binary_resource = sdk2.parameters.Resource(
            "Resource with binary",
            required=False
        )
        arcadia_binary_path = sdk2.parameters.String(
            "Path to arcadia project for binary",
            required=False
        )
        binary_name = sdk2.parameters.String(
            "Binary artifact name",
            required=True
        )
        binary_resource_name = sdk2.parameters.String(
            "Sandbox resource name for binary",
            default="ARCADIA_PROJECT",
            required=False
        )

        models_resource = sdk2.parameters.Resource(
            "Resource with catboost models",
            required=False
        )

        some_data_resource = sdk2.parameters.Resource(
            "Resource with some data",
            required=False
        )

        config_resource = sdk2.parameters.Resource(
            "Resource with service config",
            required=False
        )
        arcadia_config_path = sdk2.parameters.String(
            "Path to arcadia file for config",
            required=False
        )

        features_resource = sdk2.parameters.Resource(
            "Resource with features configs",
            required=False
        )
        custom_files = sdk2.parameters.List("Custom files from arcadia (usually used for features)")

        custom_sandbox_data = sdk2.parameters.Resource(
            "Custom sandbox resources",
            multiple=True,
            required=False
        )

    def check_params(self):
        logging.info("Checking params...")

        if not self.Parameters.binary_resource:
            if not self.Parameters.svn_revision or not self.Parameters.arcadia_binary_path:
                raise common.errors.TaskFailure('You should specify resource with binary or parameters for building binary')

        if not self.Parameters.config_resource:
            if not self.Parameters.svn_revision or not self.Parameters.arcadia_config_path:
                raise common.errors.TaskFailure('You should specify resource with config or arcadia path to config')

    def build_binary(self):
        logging.info("Start child task for building binary...")

        build_task = sdk2.Task["YA_MAKE"](
            self,
            description="Build recommender binary " + self.Parameters.binary_name,
            owner=self.owner,
            priority=self.Parameters.priority,
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia@" + str(self.Parameters.svn_revision),
            arts=os.path.join(self.Parameters.arcadia_binary_path, self.Parameters.binary_name),
            targets=self.Parameters.arcadia_binary_path,
            result_rt=self.Parameters.binary_resource_name,
            result_rd="Recommender binary",
            result_single_file=True,
            checkout_mode="auto",
            build_type="release",
            build_arch="linux",
        )
        build_task.save().enqueue()
        self.Context.build_task_id = build_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.build_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def find_binary_resource_data(self):
        if self.Parameters.binary_resource:
            return sdk2.ResourceData(self.Parameters.binary_resource)
        child = sdk2.Task[self.Context.build_task_id]
        if child.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Binary build task was finished with status {}".format(child.status))
        resource = sdk2.Resource[self.Parameters.binary_resource_name].find(task=child).first()
        return sdk2.ResourceData(resource)

    def get_config_path(self):
        if self.Parameters.config_resource:
            return str(sdk2.ResourceData(self.Parameters.config_resource).path)

        config_file_path = "./config.pbtxt"
        Arcadia.export("arcadia:/arc/trunk/arcadia/" + self.Parameters.arcadia_config_path + "@"
             + str(self.Parameters.svn_revision), config_file_path)
        return config_file_path

    def add_features(self, tar):
        if not self.Parameters.features_resource:
            return

        features_resource_data = sdk2.ResourceData(self.Parameters.features_resource)
        features_dir = "./features"
        with tarfile.open(str(features_resource_data.path), "r:gz") as features_tar:
            features_tar.extractall(features_dir)

        for f in os.listdir(features_dir):
            tar.add(os.path.join(features_dir, f), arcname=f)

    def add_custom_files(self, tar):
        if self.Parameters.features_resource:
            return

        for f in self.Parameters.custom_files:
            _, file_name = f.rsplit("/", 1)
            Arcadia.export("arcadia:/arc/trunk/arcadia/" + f + "@" + str(self.Parameters.svn_revision), "./" + file_name)
            tar.add("./" + file_name, arcname=file_name)

    def  add_custom_sandbox_data(self, tar):
        if not self.Parameters.custom_sandbox_data:
            return
        for f in self.Parameters.custom_sandbox_data:
            f_path = str(sdk2.ResourceData(f).path)
            tar.add(f_path, arcname=os.path.basename(f_path))

    def create_bundle(self):
        logging.info("Start bundle building...")

        resource = sdk2.Resource[self.Parameters.bundle_resource_name](
            self, "Recommender bundle", "bundle.tar.gz")
        resource_data = sdk2.ResourceData(resource)

        tar = tarfile.open(str(resource_data.path), 'w:gz')
        tar.add(str(self.find_binary_resource_data().path), arcname=self.Parameters.binary_name)
        if self.Parameters.models_resource:
            tar.add(str(sdk2.ResourceData(self.Parameters.models_resource).path), arcname="models.tar.gz")
        if self.Parameters.some_data_resource:
            tar.add(str(sdk2.ResourceData(self.Parameters.some_data_resource).path), arcname="some_data.tar.gz")
        tar.add(self.get_config_path(), arcname="config.pbtxt")
        self.add_features(tar)
        self.add_custom_files(tar)
        self.add_custom_sandbox_data(tar)
        tar.close()

        resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.check_params:
            self.check_params()

        with self.memoize_stage.create_subtasks:
            if not self.Parameters.binary_resource:
                self.build_binary()

        with self.memoize_stage.create_bundle:
            self.create_bundle()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
